import FDBFactory from "fake-indexeddb/lib/FDBFactory";

import { MailboxFolder } from "@bluemind/backend.mail.api";

import { LocalItemContainerValue } from "@bluemind/commons.light/core.container";
import { Action, ContainerDB } from "@bluemind/service-worker-datasource";

import { ReplicatedMailboxesDB } from "./ReplicatedMailboxesDB";

vi.mock("@bluemind/logger", () => ({
    default: {
        error: vi.fn(),
        log: vi.fn()
    }
}));

const containerUid = "containerUid";
describe("ContainerDB Interface Tests", () => {
    let mockContainerDB: ContainerDB<MailboxFolder, number>;

    const testItems: LocalItemContainerValue<MailboxFolder>[] = [
        {
            uid: "1",
            internalId: 1,
            containerUid,
            value: {}
        },
        {
            uid: "2",
            internalId: 2,
            containerUid,
            value: {}
        }
    ];

    const testIds = [1, 2];

    beforeEach(async () => {
        global.indexedDB = new FDBFactory();
        mockContainerDB = new ReplicatedMailboxesDB("containerName");
        await mockContainerDB.setSyncStatus({ version: 0, stale: false }, containerUid);
        vi.clearAllMocks();
    });

    describe("putItems", () => {
        it("Put items and set changeset", async () => {
            await mockContainerDB.putItems(testItems, containerUid);

            const items = await mockContainerDB.getAllItems(containerUid);
            const changeSet = await mockContainerDB.getChangeSet(containerUid);

            expect(changeSet[0].action).toBe(Action.UPDATE);
            expect(items.length).toBe(testItems.length);
        });
    });

    describe("putItemsAndCommit", () => {
        it("Put items without updating changeset", async () => {
            await mockContainerDB.putItemsAndCommit(testItems, containerUid);

            const items = await mockContainerDB.getAllItems(containerUid);
            const changeSet = await mockContainerDB.getChangeSet(containerUid);

            expect(changeSet.length).toBe(0);
            expect(items.length).toBe(testItems.length);
        });
    });

    describe("deleteItems", () => {
        it("Delete items and set changeset", async () => {
            await mockContainerDB.putItemsAndCommit(testItems, containerUid);

            await mockContainerDB.deleteItems(testIds, containerUid);

            const items = await mockContainerDB.getAllItems(containerUid);
            const changeSet = await mockContainerDB.getChangeSet(containerUid);
            expect(changeSet[0].action).toBe(Action.DELETE);
            expect(items.length).toBe(0);
        });
    });

    describe("deleteItemsAndCommit", () => {
        it("Delete items without updating changeset", async () => {
            await mockContainerDB.putItemsAndCommit(testItems, containerUid);

            await mockContainerDB.deleteItemsAndCommit(testIds, containerUid);

            const items = await mockContainerDB.getAllItems(containerUid);
            const changeSet = await mockContainerDB.getChangeSet(containerUid);
            expect(changeSet.length).toBe(0);
            expect(items.length).toBe(0);
        });
    });

    describe("getItems", () => {
        it("Correct items are fetch", async () => {
            await mockContainerDB.putItemsAndCommit(testItems, containerUid);
            await mockContainerDB.putItemsAndCommit(testItems, "other" + containerUid);

            const items = await mockContainerDB.getItems(testIds, containerUid);

            expect(items).toStrictEqual(testItems);
        });
    });

    describe("getAllItems", () => {
        it("Get all items", async () => {
            testItems.push({
                uid: "3",
                internalId: 3,
                containerUid,
                value: {}
            });
            await mockContainerDB.putItemsAndCommit(testItems, containerUid);

            const items = await mockContainerDB.getAllItems(containerUid);

            expect(items).toStrictEqual(testItems);
        });
    });

    describe("reset", () => {
        it("Reset items", async () => {
            await mockContainerDB.putItemsAndCommit(testItems, containerUid);

            await mockContainerDB.reset(containerUid);

            const items = await mockContainerDB.getAllItems(containerUid);
            expect(items).toStrictEqual([]);
            const sync = await mockContainerDB.getSyncStatus(containerUid);
            expect(sync).toStrictEqual(undefined);
            const changeSet = await mockContainerDB.getChangeSet(containerUid);
            expect(changeSet).toStrictEqual([]);
        });
    });
});
