import { MailboxFolder } from "@bluemind/backend.mail.api";
import { ItemValue } from "@bluemind/core.container.api";

import { setupMockDatasource, setupDatasourceError } from "../../testUtils/MockDatasource";

import MailboxFolderLocalDBProxy from "./MailboxFolderLocalDBProxy";

const mockNext = vi.fn().mockResolvedValue(undefined);

vi.mock("@bluemind/logger", () => ({
    default: {
        error: vi.fn()
    }
}));

vi.mock("../../Synchronization/SyncUtils", () => ({
    relaxedSynchronize: vi.fn(),
    isLocal: vi.fn().mockResolvedValue(true),
    isSynchronized: vi.fn().mockResolvedValue(true)
}));

const setupProxy = () => new MailboxFolderLocalDBProxy("api-key", "partition", "mailbox-uid");
describe("MailboxFolderLocalDBProxy", () => {
    let mockDb: any;
    beforeEach(() => {
        const datasource = setupMockDatasource({
            getItems: vi.fn().mockResolvedValue([{ value: { name: "a" } }] as ItemValue<MailboxFolder>[]),
            getAllItems: vi
                .fn()
                .mockResolvedValue([{ value: { name: "a" } }, { value: { name: "b" } }] as ItemValue<MailboxFolder>[])
        });
        mockDb = datasource.mockDb;
    });

    afterEach(() => {
        vi.clearAllMocks();
    });

    describe("getCompleteById()", () => {
        test("retrieves complete items", async () => {
            const proxy = setupProxy();

            const result = await proxy.getCompleteById(1);

            expect(mockDb.getItems).toHaveBeenCalledOnce();
            expect(result.value.name).toBe("a");
            expect(result).toBeDefined();
        });

        test("fallbacks to next when datasource fails", async () => {
            setupDatasourceError();
            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.getCompleteById(1);

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });
    });

    describe("all()", () => {
        test("retrieves all items", async () => {
            const proxy = setupProxy();

            const result = await proxy.all();

            expect(mockDb.getAllItems).toHaveBeenCalledOnce();
            expect(result).toHaveLength(2);
        });

        test("fallbacks to next when datasource fails", async () => {
            setupDatasourceError();
            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.all();

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });
    });

    describe("byName()", () => {
        test("retrieves items byName", async () => {
            const proxy = setupProxy();

            const result = await proxy.byName("a");

            expect(mockDb.getAllItems).toHaveBeenCalledOnce();
            expect(result.value.name).toBe("a");
        });

        test("fallbacks to next when datasource fails", async () => {
            setupDatasourceError();
            const proxy = setupProxy();
            proxy.next = mockNext;

            const result = await proxy.byName("a");

            expect(proxy.next).toHaveBeenCalledOnce();
            expect(result).toBeUndefined();
        });
    });
});
