import { openDB, DBSchema, IDBPDatabase } from "idb";

import logger from "@bluemind/logger";
import session from "@bluemind/session";

interface EnvironmentSchema extends DBSchema {
    system: {
        key: string;
        value: { key: string; value: any };
    };
}

const VERSION = 1;
// eslint-disable-next-line @typescript-eslint/no-empty-interface
interface SystemKeyConstraint<T> {}
export type SystemKey<T = any> = string & SystemKeyConstraint<T>;

interface EnvironmentDB {
    getSystemValue<T>(key: SystemKey<T>): Promise<T | undefined>;
    setSystemValue<T>(key: SystemKey<T>, value: T): Promise<void>;
    setMailboxCopyGuid(uid: string): Promise<void>;
    getMailboxCopyGuid(): Promise<string | undefined>;
    close(): Promise<void>;
}
export class EnvironmentDBImpl implements EnvironmentDB {
    db: Promise<IDBPDatabase<EnvironmentSchema>>;
    constructor(name: string) {
        this.db = this.openDB(name);
    }
    private async openDB(name: string) {
        return openDB<EnvironmentSchema>(`${name}-environnement`, VERSION, {
            upgrade(db, oldVersion) {
                logger.log(`[SW][DB] Upgrading from ${oldVersion} to ${VERSION}`);
                if (oldVersion < VERSION) {
                    logger.log("[SW][DB] Upgrading deleting existing object store");
                    for (const name of Object.values(db.objectStoreNames)) {
                        db.deleteObjectStore(name);
                    }
                }
                db.createObjectStore("system", { keyPath: "key" });
            },
            blocking: async () => {
                (await this.db).close();
                this.db = this.openDB(name);
            }
        });
    }
    async close(): Promise<void> {
        (await this.db).close();
    }
    async getSystemValue<T>(key: SystemKey<T>): Promise<T | undefined> {
        return (await (await this.db).get("system", key))?.value as T;
    }
    async setSystemValue<T>(key: SystemKey<T>, value: T): Promise<void> {
        (await this.db).put("system", { key, value });
    }
    async setMailboxCopyGuid(uid: string) {
        logger.log(`[SW][DB] Initialize environment mailboxCopyGuid to ${uid}.`);
        await (await this.db).put("system", { key: "mailboxCopyGuid", value: uid });
    }

    async getMailboxCopyGuid() {
        const data = await (await this.db).get("system", "mailboxCopyGuid");
        if (data === undefined) {
            return undefined;
        }
        return data.value;
    }
}

let implementation: EnvironmentDB | null = null;
async function instance(): Promise<EnvironmentDB> {
    if (!implementation) {
        const name = `${session.userId}`;
        implementation = new EnvironmentDBImpl(name);
    }
    return implementation;
}

session.addEventListener("change", event => {
    const { old, value } = event.detail;
    if (value.userId != old?.userId && implementation) {
        implementation?.close();
        implementation = null;
    }
});

const db: EnvironmentDB = {
    getSystemValue: key => instance().then(db => db.getSystemValue(key)),
    setSystemValue: (key, value) => instance().then(db => db.setSystemValue(key, value)),
    setMailboxCopyGuid: uid => instance().then(db => db.setMailboxCopyGuid(uid)),
    getMailboxCopyGuid: () => instance().then(db => db.getMailboxCopyGuid()),
    close: () => instance().then(db => db.close())
};

export default db;
