/* eslint-disable @typescript-eslint/no-explicit-any */
import { ChangeSet, SyncService, SyncStatus } from "@bluemind/commons.light/model/synchronization";
import { ItemValue } from "@bluemind/core.container.api";

import logger from "@bluemind/logger";
import { ContainerDB } from "@bluemind/service-worker-datasource";
import { Change } from "@bluemind/service-worker-datasource/src/ContainerDB";

import { synchronize } from "./Synchronize";

let mockUpdatedItems: any[] = [];
let mockDeletedItems: any[] = [];
let mockVersion = 0;
let stale = true;

let version = 1;
const containerUid = "uid";

vi.mock("@bluemind/commons.light/utils/service-worker", () => ({
    default: {
        postMessage: vi.fn()
    }
}));
class FakeSyncService implements SyncService<ItemValue<any>, string> {
    async resetData(containerUid: string): Promise<void> {
        return;
    }
    async updateRemote(updated: ChangeSet<string>): Promise<void> {
        await "";
    }
    async getLocalChangeSet(): Promise<ChangeSet<string>> {
        return await { updated: [], deleted: [], version: 0 };
    }
    async getRemoteChangeSet(
        version: number
    ): Promise<{ uid: string; updated: string[]; deleted: string[]; version: number } | null> {
        const ret = {
            uid: containerUid,
            updated: mockUpdatedItems,
            deleted: mockDeletedItems,
            version: mockVersion || version
        };
        return ret;
    }

    async getRemoteItems(ids: (string | number)[]): Promise<ItemValue<any>[]> {
        return ids.map(id => ({ id, value: `Mock value for ${id}` }));
    }
}

const syncServ = new FakeSyncService();

type serviceSyncClass = new (sid: string, uid: string) => FakeSyncService;

vi.mock("./SyncServiceProvider", () => {
    return {
        default: {
            register(_syncServiceClass: serviceSyncClass, _type: string) {
                return "";
            },
            get(_uid: string, _type: string) {
                return syncServ;
            }
        }
    };
});

class FakeDB implements ContainerDB<any, number> {
    async reset(containerUid: string): Promise<void> {
        //
    }
    async putItems(items: ItemValue<any>[]) {
        //
    }
    async putItemsAndCommit(items: ItemValue<any>[]) {
        //
    }
    async deleteItems(ids: number[]): Promise<void> {
        //
    }
    async deleteItemsAndCommit(ids: number[]): Promise<void> {
        //
    }
    async getItems(ids: number[]) {
        return [{ value: "" }];
    }
    async getAllItems(): Promise<ItemValue<any>[]> {
        return [{ value: "" }];
    }
    async getChangeSet(): Promise<Change<number>[]> {
        return [{ uid: 0, action: 0, containerUid }];
    }
    async commit(ids: number[]): Promise<void> {
        //
    }
    async getSyncStatus(): Promise<SyncStatus> {
        return {
            version,
            stale
        };
    }
    async setSyncStatus(syncStatus: SyncStatus) {
        version = syncStatus.version;
    }
    close(): void {
        //
    }
}
const db = new FakeDB();
vi.mock("@bluemind/service-worker-datasource", () => {
    return {
        default: {
            register(uid: string, _type: string) {
                return "";
            },
            retrieve(handlerClass: any, type: string) {
                return db;
            }
        }
    };
});

describe("Synchronize", () => {
    let getRemoteChangeSetSpy = vi.spyOn(syncServ, "getRemoteChangeSet");
    let getRemoteItemsSpy = vi.spyOn(syncServ, "getRemoteItems");
    let resetDataSpy = vi.spyOn(syncServ, "resetData");
    let logSpy: ReturnType<typeof vi.spyOn>;

    beforeEach(() => {
        logSpy = vi.spyOn(logger, "error").mockImplementation(() => {
            //
        });
        logSpy = vi.spyOn(logger, "log").mockImplementation(() => {
            //
        });
        mockUpdatedItems = [];
        mockDeletedItems = [];
        mockVersion = 0;
        version = 1;
        stale = true;
        getRemoteChangeSetSpy = vi.spyOn(syncServ, "getRemoteChangeSet");
        getRemoteItemsSpy = vi.spyOn(syncServ, "getRemoteItems");
        resetDataSpy = vi.spyOn(syncServ, "resetData");
    });
    afterEach(() => {
        mockVersion = version;
        vi.clearAllMocks();
        logSpy.mockRestore();
        getRemoteChangeSetSpy.mockRestore();
        getRemoteItemsSpy.mockRestore();
    });

    it("reset the db when local version is greater than remote", async () => {
        mockVersion = version - 2;
        const result = await synchronize({ uid: containerUid, type: "type" });

        expect(resetDataSpy).toHaveBeenCalledWith(containerUid);
        expect(result).toBe(true);
    });

    it("same version should not update nor delete locally", async () => {
        mockVersion = version;
        const result = await synchronize({ uid: containerUid, type: "type" });

        expect(getRemoteChangeSetSpy).toHaveBeenCalled();
        expect(getRemoteItemsSpy).not.toHaveBeenCalled();
        expect(result).toBe(true);
    });

    it("different version should update locally", async () => {
        mockVersion = version + 1;
        mockUpdatedItems = ["item1", "item2"];
        mockDeletedItems = ["item3"];

        const result = await synchronize({ uid: containerUid, type: "type" });

        expect(getRemoteChangeSetSpy).toHaveBeenCalled();
        expect(getRemoteItemsSpy).toHaveBeenCalledWith(mockUpdatedItems);
        expect(version).toBe(mockVersion);
        expect(result).toBe(true); // ou la valeur attendue en cas de succès
    });

    it("up to date (!stale) container, should not proceed with synchronization", async () => {
        mockUpdatedItems = ["item1", "item2"];
        mockDeletedItems = ["item3"];
        stale = false;

        const result = await synchronize({ uid: containerUid, type: "type" });

        expect(getRemoteChangeSetSpy).not.toHaveBeenCalled();
        expect(result).toBe(true); // ou la valeur attendue en cas de succès
    });

    it("should return false when getRemoteChangeSet throws an error", async () => {
        vi.spyOn(db, "setSyncStatus").mockRejectedValue(new Error("Database error"));

        const result = await synchronize({ uid: containerUid, type: "type" });

        expect(result).toBe(false);
    });
});
