package net.bluemind.keycloak.utils.sessions;

import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.auth0.jwt.JWT;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.interfaces.DecodedJWT;

import io.vertx.core.http.HttpMethod;
import io.vertx.core.json.JsonObject;
import net.bluemind.keycloak.utils.KeycloakAdminClient;
import net.bluemind.keycloak.utils.endpoint.KeycloakEndpoints;

public class JwtManager {
	private static final Logger logger = LoggerFactory.getLogger(JwtManager.class);

	private final String realm;
	@SuppressWarnings("unused")
	private final JsonObject jwtToken;
	private final DecodedJWT accessToken;

	public static Optional<JwtManager> fromJson(String realm, JsonObject jwtToken) {
		if (realm == null) {
			logger.debug("Null realm, ignoring");
			return Optional.empty();
		}

		if (jwtToken == null) {
			logger.error("Null JWTToken, ignoring");
			return Optional.empty();
		}

		String accessToken = jwtToken.getString("access_token");
		if (accessToken == null || accessToken.isBlank()) {
			logger.error("Null or empty access token: '{}', ignoring", accessToken);
			return Optional.empty();
		}

		try {
			return Optional.of(new JwtManager(realm, jwtToken, JWT.decode(accessToken)));
		} catch (JWTDecodeException jwtDE) {
			logger.error("Invalid access token: {}, ignoring", jwtDE.getMessage());
		}

		return Optional.empty();
	}

	private JwtManager(String realm, JsonObject jwtToken, DecodedJWT accessToken) {
		this.realm = realm;
		this.jwtToken = jwtToken;
		this.accessToken = accessToken;
	}

	public void logoutUserSession() {
		// Logout all session is broken:
		// https://github.com/keycloak/keycloak/issues/27359
		KeycloakAdminClient.getInstance().call(KeycloakEndpoints.sessionEndpoint(realm,
				accessToken.getClaim("sid").asString(), Map.of("isOffline", "false")), HttpMethod.DELETE);
	}
}
