/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.webmodule.project.vite.internal;

import java.io.IOException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.Optional;

import org.osgi.framework.Bundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.json.JsonObject;
import net.bluemind.webmodule.server.js.JsDependency;
import net.bluemind.webmodule.server.project.AbstractProject;

public abstract class AbstractViteResource extends AbstractProject implements ViteProject {



	private static final Logger logger = LoggerFactory.getLogger(AbstractViteResource.class);
	protected List<ManifestChunk> chunks = new ArrayList<>();
	protected List<JsDependency> dependencies = new ArrayList<>();

	public AbstractViteResource(Bundle bundle) {
		super(bundle);
		Optional<Path> rootPath = findRootPath();
		if (rootPath.isPresent()) {
			setRootDir(rootPath.get().toString());
			if (root.isPresent()) {
				try {
					JsonObject manifest = new JsonObject(Files.readString(root.get().resolve("manifest.json")));
					manifest.forEach(entry -> chunks
							.add(new ManifestChunk(entry.getKey(), manifest.getJsonObject(entry.getKey()))));
					loadResources();
				} catch (IOException e) {
					logger.error("Could not read manifest file from {} ", getBundleName(), e);
				}
			}
		} else {
			logger.error("no manifest.json found into {} ", getBundleName());
		}

	}

	private Optional<Path> findRootPath() {
		Enumeration<URL> entries = bundle.findEntries("/", "manifest.json", true);
		if (entries != null && entries.hasMoreElements()) {
			URL url = entries.nextElement();
			return Optional.of(Paths.get(url.getFile()).getParent());
		}
		return Optional.empty();
	}

	protected abstract void loadResources();

	public void addDependency(JsDependency dependency) {
		dependencies.add(dependency);
	}

}
