/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.container.model.acl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public record AclDiff(String subject, Verb oldVerb, Verb newVerb, AclStatus status) {

	public enum AclStatus {
		ADDED, REMOVED, UPDATED;
	}

	@Override
	public String toString() {
		return "AclDiff [ subject = " + subject + ", oldVerb = " + (oldVerb != null ? oldVerb.name() : "None")
				+ ", new = " + (newVerb != null ? newVerb.name() : "None") + ", status= " + status.name() + "]";
	}

	public AccessControlEntry toAce() {
		return AccessControlEntry.create(subject, newVerb);
	}

	public static List<AccessControlEntry> toAcl(List<AclDiff> aclDiff) {
		return aclDiff.stream().map(diff -> diff.toAce()).toList();
	}

	public static List<AclDiff> prepareAclDiff(String subject, Collection<AccessControlEntry> previous,
			Collection<AccessControlEntry> current) {

		List<AccessControlEntry> newAcls = filteredAcls(subject, current, previous);
		List<AccessControlEntry> oldAlcs = filteredAcls(subject, previous, current);

		List<AclDiff> diffAcls = new ArrayList<>();
		for (AccessControlEntry oldAce : oldAlcs) {
			// search same subject in new
			addToListForUpdate(diffAcls, newAcls, oldAce);
		}

		for (AccessControlEntry newAce : newAcls) {
			// search same subject in old
			addToListForUpdate(diffAcls, oldAlcs, newAce);
		}

		if (newAcls.isEmpty()) {
			addToList(diffAcls, oldAlcs, AclStatus.REMOVED);
		} else {
			addToListNoneMatch(diffAcls, oldAlcs, newAcls, AclStatus.REMOVED);
		}

		if (oldAlcs.isEmpty()) {
			addToList(diffAcls, newAcls, AclStatus.ADDED);
		} else {
			addToListNoneMatch(diffAcls, newAcls, oldAlcs, AclStatus.ADDED);
		}

		return diffAcls;
	}

	private static List<AccessControlEntry> filteredAcls(String subject, Collection<AccessControlEntry> sourceList,
			Collection<AccessControlEntry> listToCompare) {
		return sourceList.stream() //
				.filter(e -> e.verb != Verb.Visible && !listToCompare.contains(e) && !e.subject.equals(subject)) //
				.toList();
	}

	private static void addToListForUpdate(List<AclDiff> result, List<AccessControlEntry> sourceList,
			AccessControlEntry entry) {
		sourceList.stream().filter(v -> v.subject.equals(entry.subject)).findFirst()
				.ifPresent(v -> result.add(AclDiff.createAclDiffForUpdate(v, entry)));
	}

	private static void addToList(List<AclDiff> result, List<AccessControlEntry> sourceList, AclStatus status) {
		sourceList.forEach(aws -> result.add(AclDiff.createAclDiff(aws, status)));
	}

	private static void addToListNoneMatch(List<AclDiff> result, List<AccessControlEntry> sourceList,
			List<AccessControlEntry> listToCompare, AclStatus status) {
		sourceList.stream().filter(s -> listToCompare.stream().noneMatch(v -> v.subject.equals(s.subject)))
				.forEach(aws -> result.add(AclDiff.createAclDiff(aws, status)));
	}

	private static AclDiff createAclDiffForUpdate(AccessControlEntry oldAcl, AccessControlEntry newAcl) {
		return new AclDiff(oldAcl.subject, oldAcl.verb, newAcl.verb, AclStatus.UPDATED);
	}

	private static AclDiff createAclDiff(AccessControlEntry aws, AclStatus status) {
		switch (status) {
		case ADDED: {
			return new AclDiff(aws.subject, null, aws.verb, AclStatus.ADDED);
		}
		case REMOVED: {
			return new AclDiff(aws.subject, aws.verb, null, AclStatus.REMOVED);
		}
		default:
			throw new IllegalArgumentException("Unexpected value: " + status);
		}
	}
}
