export const CRITERIA_TARGETS = {
    BODY: "part.content",
    FROM: "from.email",
    SUBJECT: "subject",
    TO: "to.email",
    HEADER: "headers"
};

export const CRITERIA_MATCHERS = {
    CONTAINS: "CONTAINS",
    EXISTS: "EXISTS",
    EQUALS: "EQUALS",
    MATCHES: "MATCHES"
};

export const CRITERIA_OPERATORS = {
    OR: "OR",
    AND: "AND"
};

export const CRITERIA_TYPE = {
    CONDITION: "CONDITION",
    EXCEPTION: "EXCEPTION"
};

export const ACTIONS = {
    DELETE: { name: "MARK_AS_DELETED", isValid: Boolean },
    DELIVER: { name: "MOVE", isValid: action => action.folder },
    COPY: { name: "COPY", isValid: Boolean },
    DISCARD: { name: "DISCARD", isValid: Boolean },
    FORWARD: { name: "REDIRECT", isValid: action => action.emails?.length > 0 },
    TRANSFER: { name: "TRANSFER", isValid: action => action.emails?.length > 0 },
    READ: { name: "MARK_AS_READ", isValid: Boolean },
    STAR: { name: "MARK_AS_IMPORTANT", isValid: Boolean }
};

const ACTIONS_BY_NAME = new Map(Object.values(ACTIONS).map(action => [action.name, action]));

export const NEW_FILTER = Object.freeze({
    name: "",
    criteria: [{ isNew: true, exception: false }],
    actions: [{ isNew: true }],
    exceptions: [],
    criteriaOperator: CRITERIA_OPERATORS.AND,
    exceptionOperator: CRITERIA_OPERATORS.OR,
    terminal: false,
    active: true,
    editable: true
});

export function read(rules) {
    return rules.flatMap(rule => {
        return isManageable(rule)
            ? {
                  ...readRule(rule),
                  terminal: rule.stop,
                  editable: true
              }
            : [];
    });
}

function isManageable(rule) {
    return rule.client === "bluemind" && rule.trigger === "IN" && !rule.deferred;
}

export function readRule(rawRule) {
    const criteriaBy = groupByCriteriaType(readCriteria(rawRule.conditions));
    return {
        active: rawRule.active,
        name: rawRule.name,
        terminal: rawRule.stop ?? true,
        criteriaOperator: retrieveCriteriaOperator(rawRule.conditions),
        exceptionOperator: retrieveExceptionsOperator(rawRule.conditions),
        criteria: criteriaBy[CRITERIA_TYPE.CONDITION],
        exceptions: criteriaBy[CRITERIA_TYPE.EXCEPTION],
        actions: readActions(rawRule.actions),
        id: rawRule.id
    };
}

function groupByCriteriaType(conditions) {
    const groupedCondition = {
        [CRITERIA_TYPE.CONDITION]: [],
        [CRITERIA_TYPE.EXCEPTION]: []
    };

    conditions.forEach(condition => {
        const type = condition.exception ? CRITERIA_TYPE.EXCEPTION : CRITERIA_TYPE.CONDITION;
        groupedCondition[type].push(condition);
    });
    return groupedCondition;
}

function readCriteria(rawConditions, isException = null) {
    const criteria = [];
    rawConditions.forEach(rawCondition => {
        if (rawCondition.conditions.length) {
            criteria.push(...readCriteria(rawCondition.conditions, rawCondition.negate));
        }
        const filter = rawCondition.filter;
        if (filter) {
            criteria.push({
                target: readTarget(filter.fields[0]),
                matcher: filter.operator,
                value: filter.values && filter.values.length > 0 ? filter.values[0] : undefined,
                exception: isException ?? rawCondition.negate
            });
        }
    });
    return criteria;
}

function retrieveCriteriaOperator(conditions) {
    return retreiveOperator(conditions, CRITERIA_TYPE.CONDITION) || CRITERIA_OPERATORS.AND;
}
function retrieveExceptionsOperator(conditions) {
    return retreiveOperator(conditions, CRITERIA_TYPE.EXCEPTION) || CRITERIA_OPERATORS.OR;
}
function retreiveOperator(conditions, type) {
    const isException = type === CRITERIA_TYPE.EXCEPTION;
    const criteriaCondition = conditions.find(condition => condition.negate === isException);
    return criteriaCondition?.conditions[0]?.operator;
}

function readTarget(field) {
    const fieldTokens = field.split(".");
    return {
        type: field.startsWith("headers") ? fieldTokens[0] : field,
        name: field.startsWith("headers") ? fieldTokens[1] : undefined
    };
}

function readActions(rawActions) {
    return rawActions
        .map(rawAction => (ACTIONS_BY_NAME.get(rawAction.name)?.isValid(rawAction) ? { ...rawAction } : undefined))
        .filter(Boolean);
}

export function write(filter) {
    return {
        client: "bluemind",
        type: "GENERIC",
        trigger: "IN",
        deferred: false,
        active: filter.active,
        name: filter.name || undefined,
        conditions: [
            {
                conditions: filter.criteria.map(criterion => writeCondition(criterion, filter.criteriaOperator)),
                negate: false,
                operator: "AND"
            },
            {
                conditions: filter.exceptions.map(exception => writeCondition(exception, filter.exceptionOperator)),
                negate: true,
                operator: "AND"
            }
        ].filter(condition => condition.conditions.length),
        actions: writeActions(filter.actions),
        stop: filter.terminal
    };
}

function writeCondition(criterion, operator) {
    const field =
        criterion.target.type === "headers"
            ? criterion.target.type + "." + criterion.target.name
            : criterion.target.type;
    const fields = field === "to.email" ? [field, "cc.email"] : [field];
    const condition = {
        negate: false,
        operator: operator.toUpperCase(),
        conditions: [],
        filter: {
            fields,
            operator: criterion.matcher
        }
    };
    if (criterion.matcher !== "EXISTS") {
        condition.filter.values = criterion.value ? [criterion.value] : [];
    }
    return condition;
}

export function writeActions(actions) {
    return actions
        .map(action => (ACTIONS_BY_NAME.get(action.name)?.isValid(action) ? { ...action } : undefined))
        .filter(Boolean);
}

export function toString(filter, vueI18N) {
    const i18nInstance = typeof vueI18N === "function" ? { t: vueI18N } : vueI18N;
    function joinWithSpace(key) {
        return ` ${i18nInstance.t(`common.${key}`)} `;
    }

    const actions = filter.actions
        .map(action => {
            let value;
            if ([ACTIONS.FORWARD.name, ACTIONS.TRANSFER.name].includes(action.name)) {
                value = action.emails.join(joinWithSpace("and"));
            } else if (ACTIONS.DELIVER.name === action.name) {
                value = action.folder;
            } else {
                value = action.value;
            }
            return i18nInstance.t(`preferences.mail.filters.action.${action.name}`, { value });
        })
        .join(joinWithSpace("then"));

    const criteria = filter.criteria
        .map(criterion => {
            const target = i18nInstance.t(`preferences.mail.filters.target.${criterion.target.type}`, {
                name: criterion.target.name
            });
            const matcher = i18nInstance.t(`preferences.mail.filters.matcher.${criterion.matcher}`);
            return `${target} ${matcher} ${criterion.value}`;
        })
        .join(joinWithSpace(filter.criteriaOperator.toLowerCase()));

    const exceptions = filter.exceptions
        .map(exception => {
            const target = i18nInstance.t(`preferences.mail.filters.target.${exception.target.type}`, {
                name: exception.target.name
            });
            const matcher = i18nInstance.t(`preferences.mail.filters.matcher.${exception.matcher}.negate`);
            return `${target} ${matcher} ${exception.value}`;
        })
        .join(joinWithSpace(filter.exceptionOperator.toLowerCase()));

    const conditions = [criteria, exceptions].filter(Boolean).join(joinWithSpace("and"));

    return [actions, conditions].join(joinWithSpace("if"));
}
