import serviceWorker from "@bluemind/commons/utils/service-worker";
import { OwnerSubscriptionsClient } from "@bluemind/core.container.api";
import session from "@bluemind/session";
import WebsocketClient from "@bluemind/sockjs";
import VueBus from "@bluemind/vue-bus";

type RegisterFn = (containerUid: string) => void;

async function synchronize(uid: string, type: string) {
    if (await serviceWorker.isAvailable()) {
        //FIXME : Throttle serviceWorker.postMessage correctly to avoid losing calls
        serviceWorker.postMessage("SYNCHRONIZE", { uid, type }, { sid: session.sid });
    } else {
        const bus = new VueBus.Client();
        bus.$emit("container:updated", { uid, type });
    }
}

const RegisterFactory = {
    fn: new Map<string, RegisterFn>(),
    register(containerType: string, callback: RegisterFn) {
        this.fn.set(containerType, callback);
    },
    create(containerType: string, socket: WebsocketClient): RegisterFn {
        const register = this.fn.has(containerType)
            ? this.fn.get(containerType)
            : this.defaultRegisterFn(containerType, socket);
        return register as RegisterFn;
    },
    defaultRegisterFn(containerType: string, socket: WebsocketClient): RegisterFn {
        return containerUid => {
            const topicName = `bm.${containerType}.hook.${containerUid}.changed`;
            socket.register(topicName, synchronize.bind(null, containerUid, containerType));
        };
    }
};

export const ContainerObserver = {
    async start() {
        if (session.domain === "global.virt") {
            return;
        }
        const client = new OwnerSubscriptionsClient(session.sid, session.domain, session.userId);
        const subscriptions = await client.list();
        const socket = new WebsocketClient();
        const bus = new VueBus.Client();

        RegisterFactory.register("mailboxacl", containerUid => {
            const topic = `mailreplica.${containerUid.replace(/^mailbox:acls-/, "")}.updated`;
            socket.register(topic, ({ data }: any) => {
                if (data.body.isHierarchy) {
                    synchronize(containerUid, "mailboxacl");
                } else {
                    synchronize(data.body.mailbox.replace(/^mailbox:acls-/, ""), "mail_record");
                }
            });
        });

        subscriptions.forEach(({ value: { containerUid, containerType } }) => {
            const register = RegisterFactory.create(containerType!, socket);
            register(containerUid!);
        });

        serviceWorker.addMessageListener("UPDATED", ({ uid, type }: { uid: string; type: string }) => {
            bus.$emit("container:updated", { uid, type });
        });
    }
};
