/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.webmodule.authenticationfilter.internal;

public class Credentials {
	@SuppressWarnings("serial")
	public static class CredentialBuilderValidationException extends RuntimeException {
		public CredentialBuilderValidationException(String msg) {
			super(msg);
		}
	}

	public static class Builder {
		/**
		 * Build new credential for BlueMind domain.
		 * 
		 * @param domainUid BlueMind domain UID, must be get from a trusted source
		 * @return
		 */
		public static Builder newBuilder(String domainUid) {
			return new Builder(domainUid);
		}

		public static Builder fromExternalCred(Credentials externalCreds) {
			return new Builder(externalCreds.domainUid).setInternalAuth(externalCreds.internalAuth)
					.init(externalCreds.login, externalCreds.domain, externalCreds.userUid, externalCreds.domainUid);
		}

		private boolean internalAuth = false;
		private String login;
		private String domain;
		private String userUid;
		private String domainUid;

		public Builder(String domainUid) {
			this.domainUid = domainUid;
		}

		private Builder init(String login, String domain, String userUid, String domainUid) {
			this.login = login;
			this.domain = domain;
			this.userUid = userUid;
			this.domainUid = domainUid;
			return this;
		}

		/**
		 * 
		 * @param loginAtDomain user login@domain.tld
		 * @return
		 * @throws CredentialBuilderValidationException if loginAtDomain is invalid
		 */
		public Builder setLoginAtDomain(String loginAtDomain) {
			if (!loginAtDomain.contains("@")) {
				throw new CredentialBuilderValidationException("Invalid loginAdDomain '" + loginAtDomain + "'");
			}

			try {
				login = loginAtDomain.substring(0, loginAtDomain.indexOf("@"));
				login = login.isBlank() ? null : login;

				domain = loginAtDomain.substring(loginAtDomain.indexOf("@") + 1);
				domain = domain.isBlank() ? null : domain;
			} catch (IndexOutOfBoundsException ioobe) {
				throw new CredentialBuilderValidationException("Invalid loginAdDomain '" + loginAtDomain + "'");
			}

			return this;
		}

		public Builder setUid(String uid) {
			userUid = uid.isBlank() ? null : uid;
			return this;
		}

		public Builder setInternalAuth(boolean internalAuth) {
			this.internalAuth = internalAuth;
			return this;
		}

		/**
		 * Build externalCreds from builder data
		 * 
		 * @return
		 * @throws CredentialBuilderValidationException if login and useruid are
		 *                                              undefined
		 */
		public Credentials build() {
			if (login == null && userUid == null) {
				throw new CredentialBuilderValidationException("No login or userUid defined!");
			}

			return new Credentials(this);
		}
	}

	private final boolean internalAuth;
	private final String login;
	private final String domain;
	private final String userUid;
	private final String domainUid;

	private Credentials(Builder externalCredsBuilder) {
		this.internalAuth = externalCredsBuilder.internalAuth;
		this.login = externalCredsBuilder.login;
		this.domain = externalCredsBuilder.domain;
		this.userUid = externalCredsBuilder.userUid;
		this.domainUid = externalCredsBuilder.domainUid;
	}

	public String getDomainUid() {
		// If BlueMind internal authentication and global.virt realm then trust
		// ExternalCreds domain part otherwise use domainUid from OpenID state parameter
		return (internalAuth && domain != null && domainUid.equals("global.virt")) ? domain : domainUid;
	}

	public String getUserUid() {
		return userUid;
	}

	public String getLoginDomainPart() {
		return domain;
	}

	public String getLoginAtDomain() {
		return (login == null || domain == null) ? null : login + "@" + domain;
	}

	@Override
	public String toString() {
		String latd = getLoginAtDomain();
		return new StringBuilder().append("Credential login: ").append(latd == null ? "-" : latd).append(", user UID: ")
				.append(userUid == null ? "-" : userUid).append(", domain UID: ")
				.append(domainUid == null ? "-" : domainUid).toString();
	}
}
