/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.delivery.smtp.ndr;

import java.util.Locale;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.message.MessageImpl;

public class SendmailResponseManagement {

	private final SendmailResponse sendmailResponse;
	private final String domainAlias;
	private final Message relatedMsg;
	private final Locale locale;

	public SendmailResponseManagement(SendmailResponse sendmailResponse, String domainAlias, Message relatedMsg,
			Locale locale) {
		this.sendmailResponse = sendmailResponse;
		this.domainAlias = domainAlias;
		this.relatedMsg = relatedMsg;
		this.locale = locale;
	}

	public Optional<NdrMessage> createNdrMessage() {
		if (!sendmailResponse.isFailedResponse()) {
			return Optional.empty();
		}

		MessageImpl message = new NonDeliveryReportMessage(sendmailResponse, relatedMsg, locale).createNDRMessage();

		Mailbox rcptTo = getRcptTo();
		message.setTo(rcptTo);
		message.setFrom(new Mailbox("noreply", domainAlias));

		return Optional.of(new NdrMessage(SendmailCredentials.asAdmin0(), message));
	}

	private String getOriginalSenderAddress() {
		return sendmailResponse.isSenderDifferentFrom() ? sendmailResponse.getOriginalSender()
				: sendmailResponse.getOriginalFrom();
	}

	private org.apache.james.mime4j.dom.address.Mailbox getRcptTo() {
		String[] splitAddress = getOriginalSenderAddress().split("@");
		String toLocalPart = splitAddress[0];
		String toDomainPart = splitAddress[1];
		return new org.apache.james.mime4j.dom.address.Mailbox(toLocalPart, toLocalPart, toDomainPart);
	}

	public record NdrMessage(SendmailCredentials creds, MessageImpl message) {

		@Override
		public final String toString() {
			return "NdrMessage: creds = " + creds.loginAtDomain + ", from = "
					+ message.getFrom().getFirst().getAddress() + ", rcptTo = "
					+ message.getTo().stream().map(Object::toString).collect(Collectors.joining(","))
					+ ", msg subject = " + message.getSubject();
		}
	}
}
