import { openDB, DBSchema, IDBPDatabase, IDBPTransaction, StoreNames } from "idb";

import { MailboxFolder } from "@bluemind/backend.mail.api";
import { LocalItemContainerValue } from "@bluemind/commons.light/core.container";
import { SyncStatus } from "@bluemind/commons.light/model/synchronization";
import { isDefined } from "@bluemind/commons.light/utils/lang";
import { ItemValue } from "@bluemind/core.container.api";
import logger from "@bluemind/logger";
import { ContainerDB } from "@bluemind/service-worker-datasource";
import { Action, Change } from "@bluemind/service-worker-datasource/src/ContainerDB";

interface ContainerSchema extends DBSchema {
    items: {
        key: [number, string];
        value: LocalItemContainerValue<MailboxFolder>;
        indexes: { id: [number, string]; containerUid: string };
    };

    change_set: {
        key: [number, string];
        value: Change<number>;
        indexes: { containerUid: string };
    };

    sync_status: {
        key: string;
        value: SyncStatus;
    };
}
export class ReplicatedMailboxesDB implements ContainerDB<MailboxFolder, number> {
    dbPromise: Promise<IDBPDatabase<ContainerSchema>>;
    constructor(name: string) {
        this.dbPromise = this.openDB(name);
    }
    private async openDB(name: string): Promise<IDBPDatabase<ContainerSchema>> {
        const schemaVersion = 1;
        return openDB<ContainerSchema>(`${name}`, schemaVersion, {
            upgrade(db, oldVersion) {
                logger.log(`[SW][DB] Upgrading from ${oldVersion} to ${schemaVersion}`);
                if (oldVersion < schemaVersion) {
                    logger.log("[SW][DB] Upgrading deleting existing object store");
                    for (const name of Object.values(db.objectStoreNames)) {
                        db.deleteObjectStore(name);
                    }
                }
                const store = db.createObjectStore("items", { keyPath: ["internalId", "containerUid"] });
                store.createIndex("containerUid", "containerUid");
                db.createObjectStore("change_set", { keyPath: ["uid", "containerUid"] }).createIndex(
                    "containerUid",
                    "containerUid"
                );
                db.createObjectStore("sync_status");
            },
            blocking: async () => {
                await this.close();
                this.dbPromise = this.openDB(name);
            }
        });
    }

    async close(): Promise<void> {
        (await this.dbPromise).close();
    }

    async putItems(items: ItemValue<MailboxFolder>[], containerUid: string) {
        const tx = (await this.dbPromise).transaction(["items", "change_set", "sync_status"], "readwrite");
        for (const item of items) {
            tx.objectStore("items").put({ ...item, containerUid: containerUid });
            tx.objectStore("change_set").put({
                uid: item.internalId as number,
                containerUid: containerUid,
                action: Action.UPDATE
            });
        }

        const syncStatus = await this.getOrDefaultSyncStatus(containerUid, tx);
        syncStatus.stale = true;
        await tx.objectStore("sync_status").put(syncStatus, containerUid);

        await tx.done;
    }

    async putItemsAndCommit(items: ItemValue<MailboxFolder>[], containerUid: string) {
        const tx = (await this.dbPromise).transaction("items", "readwrite");
        for (const item of items) {
            tx.objectStore("items").put({ ...item, containerUid: containerUid });
        }
        await tx.done;
    }

    async deleteItems(
        ids: number[],
        containerUid: string,
        _tx?: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const tx = _tx || (await this.dbPromise).transaction(["items", "change_set", "sync_status"], "readwrite");
        for (const id of ids) {
            tx.objectStore("items").delete([id, containerUid]);
            tx.objectStore("change_set").put({ uid: id, containerUid: containerUid, action: Action.DELETE });
        }
        const syncStatus = await this.getOrDefaultSyncStatus(containerUid, tx);
        syncStatus.stale = true;
        await tx.objectStore("sync_status").put(syncStatus, containerUid);

        if (!_tx) {
            await tx.done;
        }
    }

    async deleteItemsAndCommit(
        ids: number[],
        containerUid: string,
        _tx?: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const tx = _tx || (await this.dbPromise).transaction(["items"], "readwrite");
        const store = tx.objectStore("items");
        for (const id of ids) {
            store.delete([id, containerUid]);
        }

        if (!_tx) {
            await tx.done;
        }
    }

    async getItems(ids: number[], containerUid: string) {
        const tx = (await this.dbPromise).transaction("items", "readonly");
        const store = tx.objectStore("items");
        return (await Promise.all(ids.map(id => store.get([id, containerUid])))).filter(isDefined);
    }

    async getAllItems(containerUid: string) {
        return (await this.dbPromise).getAllFromIndex("items", "containerUid", containerUid);
    }
    async getChangeSet(containerUid: string) {
        return (await this.dbPromise).getAllFromIndex("change_set", "containerUid", containerUid);
    }

    async commit(ids: number[], containerUid: string) {
        const tx = (await this.dbPromise).transaction("change_set", "readwrite");
        for (const id of ids) {
            tx.objectStore("change_set").delete([id, containerUid]);
        }
        await tx.done;
    }

    async getSyncStatus(containerUid: string) {
        return (await this.dbPromise).get("sync_status", containerUid);
    }
    async setSyncStatus(syncStatus: SyncStatus, containerUid: string) {
        (await this.dbPromise).put("sync_status", syncStatus, containerUid);
    }

    private async getOrDefaultSyncStatus(
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], IDBTransactionMode>
    ) {
        return (
            (await tx.objectStore("sync_status").get(containerUid)) || {
                stale: false,
                version: 0
            }
        );
    }

    async reset(containerUid: string): Promise<void> {
        const tx = (await this.dbPromise).transaction(["items", "change_set", "sync_status"], "readwrite");
        await Promise.all([
            this.resetItems(containerUid, tx),
            this.resetChangeSet(containerUid, tx),
            this.resetSyncStatus(containerUid, tx)
        ]);
        await tx.done;
    }

    private async resetItems(
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const items = await tx.objectStore("items").index("containerUid").getAll(containerUid);

        const internalIds = items.flatMap(item => item.internalId) || [];
        await this.deleteItemsAndCommit(internalIds as number[], containerUid, tx);
    }

    private async resetChangeSet(
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        const changeSetStore = tx.objectStore("change_set");
        const changeSetKeys = await changeSetStore.index("containerUid").getAllKeys(containerUid);
        await Promise.all(changeSetKeys.map(key => changeSetStore.delete(key)));
    }

    private async resetSyncStatus(
        containerUid: string,
        tx: IDBPTransaction<ContainerSchema, StoreNames<ContainerSchema>[], "readwrite">
    ) {
        tx.objectStore("sync_status").delete(containerUid);
    }
}
