import { describe, it, expect, vi, beforeEach } from "vitest";

import { MailboxFolder, MailboxFoldersByOwnerClient } from "@bluemind/backend.mail.api";
import { ItemValue } from "@bluemind/core.container.api";
import bus from "@bluemind/service-worker-bus/src/bus";
import ContainerDatasource, { Action } from "@bluemind/service-worker-datasource";

import MailboxSyncService from "./MailboxSyncService";

vi.mock("@bluemind/backend.mail.api", () => {
    return {
        MailboxFoldersByOwnerClient: vi.fn().mockImplementation(() => ({
            changeset: vi.fn(),
            deleteById: vi.fn(),
            updateById: vi.fn(),
            getCompleteById: vi.fn(),
            changesetById: vi.fn()
        }))
    };
});

vi.mock("@bluemind/service-worker-datasource", () => {
    return {
        default: {
            retrieve: vi.fn()
        },
        Action: {
            CREATE: 0,
            UPDATE: 1,
            DELETE: 2
        }
    };
});

vi.mock("@bluemind/logger", () => {
    return {
        default: {
            log: vi.fn()
        }
    };
});

vi.mock("@bluemind/service-worker-bus/src/bus", () => ({
    default: {
        channel: vi.fn(() => ({
            emit: vi.fn(),
            on: vi.fn(),
            once: vi.fn(),
            off: vi.fn(),
            reset: vi.fn()
        }))
    }
}));
describe("MailboxSyncService", () => {
    let service: MailboxSyncService;
    let mockClient: MailboxFoldersByOwnerClient;
    const mockEmit = vi.fn();
    let mockDB: {
        getItems: ReturnType<typeof vi.fn>;
        getAllItems: ReturnType<typeof vi.fn>;
        getChangeSet: ReturnType<typeof vi.fn>;
        commit: ReturnType<typeof vi.fn>;
        reset: ReturnType<typeof vi.fn>;
    };

    beforeEach(() => {
        vi.clearAllMocks();

        mockDB = {
            getItems: vi.fn(),
            getAllItems: vi.fn().mockResolvedValue([{ uid: "uid" }]),
            getChangeSet: vi.fn(),
            commit: vi.fn(),
            reset: vi.fn()
        };

        (ContainerDatasource.retrieve as any).mockReturnValueOnce(mockDB);

        service = new MailboxSyncService("test-uid", "test-sid", "test-domain");
        mockClient = service.client;

        vi.mocked(bus.channel).mockReturnValue({
            emit: mockEmit
        } as any);
    });

    describe("constructor", () => {
        it("should initialize with the correct parameters", () => {
            expect(MailboxFoldersByOwnerClient).toHaveBeenCalledWith("test-sid", "test-domain", "test-uid");
            expect(ContainerDatasource.retrieve).toHaveBeenCalledWith("mailboxacl");
            expect(service.client).toBeDefined();
            expect(service.db).toBeDefined();
        });
    });
    describe("getRemoteChangeSet", () => {
        it("should fetch remote changes since a given version", async () => {
            const mockChangesetById = {
                created: [3, 4],
                updated: [1, 2],
                deleted: [5, 6],
                version: 1
            };

            vi.mocked(mockClient.changesetById).mockResolvedValue(mockChangesetById);

            const result = await service.getRemoteChangeSet(2);

            expect(result).toEqual({
                updated: [3, 4, 1, 2],
                deleted: [5, 6],
                version: 1
            });
        });
    });

    describe("getRemoteItems", () => {
        it("should fetch remote items", async () => {
            const ids = [1, 2];
            const mockResponse1: ItemValue<MailboxFolder> = { uid: "id1", value: {} };

            vi.mocked(mockClient.getCompleteById).mockResolvedValueOnce(mockResponse1);
            vi.mocked(mockClient.getCompleteById).mockResolvedValueOnce(null as unknown as ItemValue<MailboxFolder>);

            const result = await service.getRemoteItems(ids);

            expect(mockClient.getCompleteById).toHaveBeenCalledTimes(2);
            expect(result).toEqual([mockResponse1]);
            expect(mockEmit).toHaveBeenCalledTimes(1);
        });

        it("should handle empty id list", async () => {
            const result = await service.getRemoteItems([]);

            expect(mockClient.getCompleteById).not.toHaveBeenCalled();
            expect(result).toEqual([]);
        });
    });

    describe("updateRemote", () => {
        it("should update remote with local changes", async () => {
            const changeSet = {
                updated: [1, 2],
                deleted: [3, 4],
                version: 0
            };

            const mockItems = [
                { uid: 1, value: "value1", version: 1 },
                { uid: 2, value: "value2", version: 1 }
            ];

            mockDB.getItems.mockImplementation(ids => {
                return mockItems.filter(item => ids.includes(item.uid));
            });

            vi.mocked(mockClient.updateById).mockResolvedValue({ version: 0, timestamp: 0 });

            await service.updateRemote(changeSet, "test-uid");

            expect(mockDB.getItems).toHaveBeenNthCalledWith(1, [changeSet.updated[0]], "test-uid");
            expect(mockDB.getItems).toHaveBeenNthCalledWith(2, [changeSet.updated[1]], "test-uid");

            expect(mockClient.updateById).toHaveBeenCalledWith(1, "value1");
            expect(mockClient.updateById).toHaveBeenCalledWith(2, "value2");
            expect(mockClient.deleteById).toHaveBeenCalledWith(3);
            expect(mockClient.deleteById).toHaveBeenCalledWith(4);
        });
    });

    describe("getLocalChangeSet", () => {
        it("should get local changes and categorize them by action type", async () => {
            const mockChangeLog = [
                { action: Action.UPDATE, uid: 3 },
                { action: Action.DELETE, uid: 4 },
                { action: Action.DELETE, uid: 5 }
            ];

            mockDB.getChangeSet.mockResolvedValue(mockChangeLog);

            const result = await service.getLocalChangeSet("test-uid");

            expect(mockDB.getChangeSet).toHaveBeenCalled();
            expect(result).toEqual({
                updated: [3],
                deleted: [4, 5],
                version: 0
            });
        });
    });

    it("Reset DB", async () => {
        const mockRecordDB = {
            reset: vi.fn()
        };
        (ContainerDatasource.retrieve as any).mockReturnValueOnce(mockRecordDB);

        await service.resetData("test");

        expect(mockDB.reset).toHaveBeenCalledWith("test");
        expect(mockRecordDB.reset).toHaveBeenCalledWith("uid");
    });
});
