import { chunk } from "lodash";

import { MailboxItem, MailboxItemsClient } from "@bluemind/backend.mail.api";
import { ChangeSet, SyncService } from "@bluemind/commons.light/model/synchronization";
import { ContainerChangeset, ItemFlag, ItemValue, type ItemVersion } from "@bluemind/core.container.api";
import ContainerDatasource, { type ContainerDB } from "@bluemind/service-worker-datasource";

export default class MailboxItemSyncService implements SyncService<MailboxItem, number> {
    client: MailboxItemsClient;
    db: Promise<ContainerDB<MailboxItem, number>>;

    constructor(uid: string, sid: string) {
        this.client = new MailboxItemsClient(sid, uid);
        this.db = ContainerDatasource.retrieve<MailboxItem, number>("mail_record");
    }
    async getRemoteChangeSet(since: number) {
        const { created, updated, deleted, version } = (await this.client.filteredChangesetById(since, {
            must: [],
            mustNot: [ItemFlag.Deleted]
        })) as Required<ContainerChangeset<ItemVersion>>;
        return {
            updated: created.concat(updated).map(({ id }) => id!),
            deleted: deleted.map(({ id }) => id!),
            version
        };
    }
    async getRemoteItems(ids: number[]) {
        const chunks = [...chunk(ids, 200)];
        const responses = await Promise.all(chunks.map(chunk => this.client.multipleGetById(chunk)));
        const result: ItemValue<MailboxItem>[] = [];
        return result.concat(...responses).filter(Boolean);
    }
    async getLocalChangeSet(containerUid: string) {
        const changeLog = await (await this.db).getChangeSet(containerUid);
        const { created, updated, deleted } = changeLog.reduce<{
            created: number[];
            updated: number[];
            deleted: number[];
        }>(
            (result, item) => {
                const { action, uid } = item;
                switch (action) {
                    case 0:
                        result.created?.push(uid);
                        break;
                    case 1:
                        result.updated?.push(uid);
                        break;
                    case 2:
                        result.deleted?.push(uid);
                        break;
                }

                return result;
            },
            { created: [], updated: [], deleted: [] }
        );
        return { created, updated, deleted, version: 0 };
    }

    async updateRemote({ updated, deleted }: ChangeSet<number>, containerUid: string) {
        const db = await this.db;

        await Promise.all(
            updated.map(async id => {
                const itemValue = (await db.getItems([id], containerUid))[0];
                return this.client.updateById(id, itemValue.value);
            })
        );
        await this.client.multipleDeleteById(deleted);

        await db.commit(updated.concat(deleted), containerUid);
    }

    async resetData(containerUid: string) {
        await (await this.db).reset(containerUid);
    }
}
