import { describe, it, expect, vi, beforeEach, afterEach } from "vitest";

import { ItemFlag } from "@bluemind/core.container.api";

import containerDatasource, { type ContainerDB } from "@bluemind/service-worker-datasource/";

import { setupMockDatasource } from "../testUtils/MockDatasource";

import { synchronize } from "./Synchronize";
import SynchronizeQueue from "./SynchronizeQueue";
import { isSynchronized, isLocal, relaxedSynchronize, key, nudgeSync } from "./SyncUtils";

vi.mock("./Synchronize");
vi.mock("./SynchronizeQueue");

const mockSynchronize = vi.mocked(synchronize);
const mockSynchronizeQueue = vi.mocked(SynchronizeQueue);

const TEST_DATA = {
    userId: "user123",
    domainUid: "domain.com",
    containerUid: "container123",
    containerType: "container"
} as const;

const createSubscription = (containerType = TEST_DATA.containerType) => ({
    uid: `sub-of-${TEST_DATA.userId}-to-${TEST_DATA.containerUid}`,
    value: { containerType }
});

const mockSession = () => {
    vi.mock("@bluemind/session", () => ({
        default: {
            userId: vi.fn().mockReturnValue("userId"),
            domain: vi.fn().mockReturnValue("domain"),
            addEventListener: vi.fn()
        }
    }));
};

describe("Synchronization Module", () => {
    let mockContainerDatasource: typeof containerDatasource;
    let mockedDB: ContainerDB<unknown, any>;
    beforeEach(mockSession);
    afterEach(() => vi.clearAllMocks());

    describe("key constant", () => {
        it("returns correct event bus identifier", () => {
            expect(key).toBe("synchronization");
        });
    });

    const setupMocks = (subscription: any, syncStatus: any, mockDatasource: {} = {}) => {
        const mockDB: { getItems: () => {}; getSyncStatus?: () => {} } = {
            getItems: vi.fn().mockResolvedValue(subscription ? [subscription] : [])
        };

        if (syncStatus !== undefined) {
            mockDB.getSyncStatus = vi.fn().mockResolvedValue(syncStatus);
        }
        ({ mockContainerDatasource, mockDb: mockedDB } = setupMockDatasource(mockDB, mockDatasource));
    };

    describe("isLocal", () => {
        it("returns true when container has sync status", async () => {
            const syncStatus = { version: 1, lastSync: Date.now() };
            setupMocks(undefined, syncStatus);

            const result = await isLocal(TEST_DATA.containerUid, TEST_DATA.containerType);

            expect(result).toBe(true);
        });

        it("returns false when container has no sync status", async () => {
            setupMocks(undefined, undefined);

            const result = await isLocal(TEST_DATA.containerUid, TEST_DATA.containerType);

            expect(result).toBe(false);
        });

        it("returns false if the filter is not synced", async () => {
            const syncStatus = { version: 1, lastSync: Date.now() };
            setupMocks(undefined, syncStatus);
            const filter = { must: [], mustNot: [] };

            const result = await isLocal("uid1", "type1", filter);

            expect(result).toBe(false);
        });

        it("returns true if the filter is synced ", async () => {
            const syncStatus = { version: 1, lastSync: Date.now() };
            setupMocks(undefined, syncStatus);
            const filter = { must: [], mustNot: [ItemFlag.Deleted] };

            const result = await isLocal("uid2", "type2", filter);

            expect(result).toBe(true);
        });
    });

    describe("isSynchronized", () => {
        it("returns true when container is not in queue ", async () => {
            mockSynchronizeQueue.isInQueue.mockReturnValue(false);

            const result = isSynchronized(TEST_DATA.containerUid, TEST_DATA.containerType);

            expect(result).toBe(true);
        });

        it("returns false when container is in queue", async () => {
            mockSynchronizeQueue.isInQueue.mockReturnValue(true);

            const result = isSynchronized(TEST_DATA.containerUid, TEST_DATA.containerType);

            expect(result).toBe(false);
        });
    });
    describe("relaxedSynchronize", () => {
        it("calls synchronize when database and sync status exist", async () => {
            const syncStatus = { version: 1, lastSync: Date.now() };
            setupMocks(createSubscription(), syncStatus);

            await relaxedSynchronize(TEST_DATA.containerUid, TEST_DATA.containerType);

            expect(mockSynchronize).toHaveBeenCalledWith({
                uid: TEST_DATA.containerUid,
                type: TEST_DATA.containerType
            });
        });

        it("does not call synchronize when sync status is undefined", async () => {
            setupMocks(undefined, undefined);

            await relaxedSynchronize(TEST_DATA.containerUid, TEST_DATA.containerType);

            expect(mockSynchronize).not.toHaveBeenCalled();
        });

        it("calls synchronize with custom type when sync status exists", async () => {
            const syncStatus = { version: 1, lastSync: Date.now() };
            const customType = "customType";
            setupMocks(undefined, syncStatus);

            await relaxedSynchronize(TEST_DATA.containerUid, customType);

            expect(mockSynchronize).toHaveBeenCalledWith({
                uid: TEST_DATA.containerUid,
                type: customType
            });
        });
    });
    describe("nudgeSync", () => {
        it("calls flush one when nudgeSync called", async () => {
            const syncStatus = { version: 1, lastSync: Date.now() };
            setupMocks(createSubscription(), syncStatus);

            await nudgeSync(TEST_DATA.containerUid, TEST_DATA.containerType);

            expect(mockSynchronizeQueue.flushOne).toHaveBeenCalledWith({
                uid: TEST_DATA.containerUid,
                type: TEST_DATA.containerType
            });
        });
    });
});
