import pLimit from "p-limit";
import { Limit } from "p-limit";

import type {
    ContainerSyncOptions,
    ItemIdType,
    SyncService,
    SyncStatus
} from "@bluemind/commons.light/model/synchronization";
import serviceWorker from "@bluemind/commons.light/utils/service-worker";
import logger from "@bluemind/logger";
import ContainerDatasource, { type ContainerDB } from "@bluemind/service-worker-datasource";

import SynchronizeQueue from "./SynchronizeQueue";
import SyncServiceProvider from "./SyncServiceProvider";

export async function synchronize(container: ContainerSyncOptions): Promise<boolean> {
    if (container.uid.endsWith("@global.virt.subscriptions")) {
        return false;
    }
    logger.log(`[Synchronization] synchronize ${container.uid}`);
    try {
        const syncServiceClient = await SyncServiceProvider.get(container.uid, container.type);

        if (syncServiceClient === undefined) {
            logger.log(`[Synchronization] ${container.type} has no SyncProvider, ${container.uid} will not be synced `);
            return false;
        }
        const db = await ContainerDatasource.retrieve(container.type);
        await flagAsStale(container.uid, db);
        const updated = await limit(container.uid, async () => {
            const { stale, version } = (await db.getSyncStatus(container.uid)) as SyncStatus;
            if (stale) {
                return await synchronizeContainerToVersion(syncServiceClient, db, container, version);
            }
        });
        if (updated) {
            serviceWorker.postMessage("UPDATED", container);
        }
        return true;
    } catch (error) {
        logger.error(`[Synchronization] Fail to synchronize ${container?.uid}`, error);
        return false;
    }
}

async function synchronizeContainerToVersion<T, K extends ItemIdType>(
    syncServiceClient: SyncService<T, K>, // FIXME : The identifier type must be review
    db: ContainerDB<T, K>, // FIXME : The identifier type must be review
    { uid, type }: ContainerSyncOptions,
    version: number
): Promise<boolean> {
    try {
        const localChangeSet = await syncServiceClient.getLocalChangeSet(uid);
        if (hasLocalChange(localChangeSet)) {
            await syncServiceClient.updateRemote(localChangeSet, uid);
        }

        const changeSet = await syncServiceClient.getRemoteChangeSet(version);
        await flagAsUpToDate(uid, db, changeSet?.version);
        if (!changeSet || changeSet.version === version) return false;

        if (changeSet.version > version) {
            const itemsUpdated = await syncServiceClient.getRemoteItems(changeSet.updated);

            await db.putItemsAndCommit(itemsUpdated, uid);
            await db.deleteItemsAndCommit(changeSet.deleted, uid);

            logger.log(`[Synchronization] ${uid} has been synced`);
            return true;
        } else {
            logger.warn(`[Synchronization] The local version is greater than the remote, ${uid}'s data will be reset`);
            await syncServiceClient.resetData(uid);
            SynchronizeQueue.queue({ uid, type });

            return true;
        }
    } catch (error) {
        logger.error(`[Synchronization] error while syncing changeset for ${uid}`, error);
        await flagAsStale(uid, db, version);
        return false;
    }
}

const limits: { [uid: string]: Limit } = {};
export function limit<T>(uid: string, fn: () => Promise<T>): Promise<T> {
    if (!(uid in limits)) {
        limits[uid] = pLimit(1);
    }
    return limits[uid](fn);
}

async function flagAsStale<T, K extends ItemIdType>(
    containerUid: string,
    db: ContainerDB<T, K>,
    updatedVersion?: number
) {
    const version = updatedVersion || (await db.getSyncStatus(containerUid))?.version || 0;
    await db.setSyncStatus({ version, stale: true }, containerUid);
}

async function flagAsUpToDate<T, K extends ItemIdType>(
    containerUid: string,
    db: ContainerDB<T, K>,
    updatedVersion?: number
) {
    const version = updatedVersion || (await db.getSyncStatus(containerUid))?.version || 0;
    await db.setSyncStatus({ version, stale: false }, containerUid);
}

function hasLocalChange(changeSet: { updated: (string | number)[]; deleted: (string | number)[] }) {
    return changeSet.updated.length > 0 || changeSet.deleted.length > 0;
}
