/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.addressbook.adapter;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collections;
import java.util.List;

import org.apache.commons.codec.DecoderException;
import org.jsoup.Jsoup;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.api.VCard.Communications;
import net.bluemind.addressbook.api.VCard.Communications.Tel;
import net.bluemind.addressbook.api.VCard.DeliveryAddressing;
import net.bluemind.lib.ical4j.vcard.property.AddressbookServerKind;
import net.bluemind.lib.ical4j.vcard.property.AddressbookServerMember;
import net.bluemind.lib.ical4j.vcard.property.NoteAsHtml;
import net.fortuna.ical4j.model.Date;
import net.fortuna.ical4j.vcard.Property;
import net.fortuna.ical4j.vcard.property.Address;
import net.fortuna.ical4j.vcard.property.BDay;
import net.fortuna.ical4j.vcard.property.Categories;
import net.fortuna.ical4j.vcard.property.Email;
import net.fortuna.ical4j.vcard.property.Fn;
import net.fortuna.ical4j.vcard.property.Gender;
import net.fortuna.ical4j.vcard.property.Impp;
import net.fortuna.ical4j.vcard.property.Key;
import net.fortuna.ical4j.vcard.property.Kind;
import net.fortuna.ical4j.vcard.property.Member;
import net.fortuna.ical4j.vcard.property.N;
import net.fortuna.ical4j.vcard.property.Nickname;
import net.fortuna.ical4j.vcard.property.Note;
import net.fortuna.ical4j.vcard.property.Org;
import net.fortuna.ical4j.vcard.property.Role;
import net.fortuna.ical4j.vcard.property.Telephone;
import net.fortuna.ical4j.vcard.property.Title;
import net.fortuna.ical4j.vcard.property.Url;

/**
 * Adapter from {@link VCard} (BM api) to {@link net.fortuna.ical4j.vcard.VCard}
 * (ical4j)
 */
public class VCardAdapterFromBm {

	private static final Logger LOGGER = LoggerFactory.getLogger(VCardAdapterFromBm.class);
	private static final String containerSeparator = "#";

	private final VCard vcard;

	private net.fortuna.ical4j.vcard.VCard ret;
	private List<Property> properties;

	public VCardAdapterFromBm(VCard vcard) {
		this.vcard = vcard;
		this.ret = new net.fortuna.ical4j.vcard.VCard();
		this.properties = ret.getProperties();
	}

	public net.fortuna.ical4j.vcard.VCard getRet() {
		return ret;
	}

	public void add(Property prop) {
		properties.add(prop);
	}

	public void addKind(VCardVersion version) {
		switch (vcard.kind) {
		case group:
			if (version == VCardVersion.v4) {
				add(new Kind("group"));
			} else {
				add(new AddressbookServerKind("group"));
			}
			break;
		case individual:
			add(new Kind("individual"));
			break;
		case org:
			add(new Kind("org"));
			break;
		default:
			throw new IllegalArgumentException("doesnt support kind [" + vcard.kind + "]");
		}
	}

	public void addIdentification() {
		addFormattedName();
		addName();
		addNickname();
		addBirthday();
		addGender();
	}

	private void addFormattedName() {
		if (vcard.identification.formatedName != null) {
			add(new Fn(vcard.identification.formatedName.value));
		}
	}

	private void addName() {
		if (vcard.identification.name != null) {
			add(new N(VCardAdapterToBm.toVCard(vcard.identification.name.parameters),
					valueOrEmpty(vcard.identification.name.familyNames) + ";"
							+ valueOrEmpty(vcard.identification.name.givenNames) + ";"
							+ valueOrEmpty(vcard.identification.name.additionalNames) + ";"
							+ valueOrEmpty(vcard.identification.name.prefixes) + ";"
							+ valueOrEmpty(vcard.identification.name.suffixes)));
		}
	}

	private void addNickname() {
		if (vcard.identification.nickname != null && !Strings.isNullOrEmpty(vcard.identification.nickname.value)) {
			add(new Nickname(VCardAdapterToBm.toVCard(vcard.identification.nickname.parameters),
					vcard.identification.nickname.value));
		}
	}

	private void addBirthday() {
		if (vcard.identification.birthday != null) {
			add(new BDay(new Date(vcard.identification.birthday.getTime())));
		}

	}

	private void addGender() {
		if (vcard.identification.gender != null && !Strings.isNullOrEmpty(vcard.identification.gender.value)) {
			add(new Gender(vcard.identification.gender.value));
		}
	}

	public void addDeliveringAddressing() {
		for (DeliveryAddressing da : vcard.deliveryAddressing) {
			if (da.address != null && !isEmpty(da)) {
				add(new Address(da.address.postOfficeBox, da.address.extentedAddress, da.address.streetAddress,
						da.address.locality, da.address.region, da.address.postalCode, da.address.countryName));

			}
		}
	}

	public void addCategories() {
		if (vcard.explanatory.categories != null && !vcard.explanatory.categories.isEmpty()) {
			add(new Categories(vcard.explanatory.categories.stream().map(c -> c.label).toArray(String[]::new)));
		}
	}

	public void addCommunications() {
		addTels();
		addEmails();
		addImpps();
	}

	private void addTels() {
		for (Tel tel : vcard.communications.tels) {
			try {
				add(new Telephone(VCardAdapterToBm.toVCard(tel.parameters), tel.value));
			} catch (URISyntaxException e) {
				LOGGER.warn("error during vcard export", e);
			}
		}
	}

	private void addEmails() {
		for (Communications.Email mail : vcard.communications.emails) {
			add(new Email(VCardAdapterToBm.toVCard(mail.parameters), mail.value));
		}
	}

	private void addImpps() {
		for (Communications.Impp impp : vcard.communications.impps) {
			try {
				add(new Impp(VCardAdapterToBm.toVCard(impp.parameters), impp.value));
			} catch (URISyntaxException e) {
				LOGGER.warn("error during vcard export", e);
			}
		}
	}

	public void addOrganization() {
		addRole();
		addTitle();
		addOrg();
	}

	private void addRole() {
		if (!Strings.isNullOrEmpty(vcard.organizational.role)) {
			add(new Role(vcard.organizational.role));
		}
	}

	private void addTitle() {

		if (!Strings.isNullOrEmpty(vcard.organizational.title)) {
			add(new Title(vcard.organizational.title));
		}
	}

	private void addOrg() {
		if (vcard.organizational.org != null && !isEmpty(vcard.organizational.org)) {
			String[] values = { //
					vcard.organizational.org.company != null ? vcard.organizational.org.company : "", //
					vcard.organizational.org.division != null ? vcard.organizational.org.division : "", //
					vcard.organizational.org.department != null ? vcard.organizational.org.department : "" };
			add(new Org(values));
		}
	}

	public void addExplanatory() {
		addUrls();
		addNote();
	}

	private void addUrls() {
		for (VCard.Explanatory.Url url : vcard.explanatory.urls) {
			try {
				add(new Url(VCardAdapterToBm.toVCard(url.parameters), url.value));
			} catch (URISyntaxException e) {
				LOGGER.warn(e.getMessage());
			}
		}
	}

	private void addNote() {
		if (!Strings.isNullOrEmpty(vcard.explanatory.note)) {
			String noteAsPlainText = Jsoup.parse(vcard.explanatory.note).text();
			if (!vcard.explanatory.note.equals(noteAsPlainText)) {
				properties.add(new NoteAsHtml(vcard.explanatory.note));
				properties.add(new Note(noteAsPlainText));
			} else {
				properties.add(new Note(vcard.explanatory.note));
			}
		}
	}

	public void addSecurityKeys() {
		for (VCard.Security.Key key : vcard.security.keys) {
			try {
				String b64 = Base64.getEncoder().encodeToString(key.value.getBytes());
				add(new Key(VCardAdapterToBm.toVCard(key.parameters), b64));
			} catch (URISyntaxException | DecoderException e) {
				LOGGER.warn(e.getMessage());
			}
		}
	}

	public void addGroupMembers(VCardVersion version, String containerUid) {
		for (VCard.Organizational.Member m : vcard.organizational.member) {
			if (version == VCardVersion.v3) {
				if (m.itemUid != null) {
					String exportedUid = getExportUid(containerUid, m);
					add(new AddressbookServerMember(getMemberParams(m), "urn:uuid:" + exportedUid));
				} else if (m.mailto != null) {
					add(new AddressbookServerMember("urn:uuid:" + m.mailto));
				} else {
					LOGGER.warn("member doesnt have uri or mailto");
				}
			} else {
				try {
					if (m.itemUid != null) {
						String exportedUid = getExportUid(containerUid, m);
						add(new Member(getMemberParams(m), exportedUid));
					} else if (m.mailto != null) {
						add(new Member(Collections.<net.fortuna.ical4j.vcard.Parameter>emptyList(),
								"urn:uuid:" + m.mailto));
					} else {
						LOGGER.warn("member doesnt have uri or mailto");
					}
				} catch (URISyntaxException e) {
					LOGGER.warn("Invalid uri: " + e.getMessage(), e);
				}
			}
		}
	}

	private static String valueOrEmpty(String value) {
		if (value == null)
			return "";
		else
			return value;
	}

	private static boolean isEmpty(net.bluemind.addressbook.api.VCard.Organizational.Org org) {
		if (!Strings.isNullOrEmpty(org.company) //
				|| !Strings.isNullOrEmpty(org.department) //
				|| !Strings.isNullOrEmpty(org.division)) {
			return false;
		}
		return true;
	}

	private static boolean isEmpty(DeliveryAddressing da) {
		if (da.address != null) {
			net.bluemind.addressbook.api.VCard.DeliveryAddressing.Address address = da.address;
			if (!Strings.isNullOrEmpty(address.value) //
					|| !Strings.isNullOrEmpty(address.postOfficeBox) //
					|| !Strings.isNullOrEmpty(address.extentedAddress) //
					|| !Strings.isNullOrEmpty(address.streetAddress) //
					|| !Strings.isNullOrEmpty(address.locality) //
					|| !Strings.isNullOrEmpty(address.region) //
					|| !Strings.isNullOrEmpty(address.postalCode) //
					|| !Strings.isNullOrEmpty(address.countryName) //
					|| (address.parameters != null && !address.parameters.isEmpty())) {
				return false;
			}
		}
		return true;
	}

	@SuppressWarnings("serial")
	private static List<net.fortuna.ical4j.vcard.Parameter> getMemberParams(VCard.Organizational.Member member) {
		List<net.fortuna.ical4j.vcard.Parameter> params = new ArrayList<>();
		if (member.commonName != null) {
			params.add(new net.fortuna.ical4j.vcard.Parameter("CN") {

				@Override
				public String getValue() {
					return member.commonName;
				}

			});
		}
		if (member.mailto != null) {
			params.add(new net.fortuna.ical4j.vcard.Parameter("MAILTO") {

				@Override
				public String getValue() {
					return member.mailto;
				}

			});

		}
		return params;
	}

	private static String getExportUid(String containerUid, VCard.Organizational.Member member) {
		return member.containerUid == null || member.containerUid.equals(containerUid) ? member.itemUid
				: member.containerUid + containerSeparator + member.itemUid;
	}

}
