/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.lib.vertx.utils;

import java.io.IOException;
import java.util.concurrent.CompletableFuture;

import io.vertx.core.AsyncResult;
import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.buffer.impl.BufferImpl;
import io.vertx.core.streams.WriteStream;
import net.bluemind.lib.vertx.internal.Result;
import net.bluemind.memory.pool.api.CommonMemoryPool;
import net.bluemind.memory.pool.api.IChunk;
import net.bluemind.memory.pool.api.IWritableChunk;

public class ChunkWriteStream implements WriteStream<Buffer> {

	private final CompletableFuture<IChunk> future;

	private final IWritableChunk chunk;

	public ChunkWriteStream(int capacity) throws IOException {
		this(CommonMemoryPool.getDefault().allocateEmpty(capacity));
	}

	public ChunkWriteStream(IWritableChunk c) {
		this.chunk = c;
		this.future = new CompletableFuture<>();
	}

	public CompletableFuture<IChunk> chunk() {
		return future;
	}

	@Override
	public WriteStream<Buffer> exceptionHandler(Handler<Throwable> handler) {
		return this;
	}

	@Override
	public Future<Void> write(Buffer data) {
		chunk.append(((BufferImpl) data).getByteBuf());
		return Future.succeededFuture();
	}

	@Override
	public void write(Buffer data, Handler<AsyncResult<Void>> handler) {
		write(data);
		handler.handle(new Result<>());
	}

	@Override
	public Future<Void> end() {
		future.complete(chunk);
		return Future.succeededFuture();
	}

	public void forceRelease() {
		chunk.release();
	}

	@Override
	public void end(Handler<AsyncResult<Void>> handler) {
		end();
		handler.handle(new net.bluemind.lib.vertx.internal.Result<>());
	}

	@Override
	public WriteStream<Buffer> setWriteQueueMaxSize(int maxSize) {
		return this;
	}

	@Override
	public boolean writeQueueFull() {
		return false;
	}

	@Override
	public WriteStream<Buffer> drainHandler(Handler<Void> handler) {
		return this;
	}

}
