/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.webmodule.server;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.StringJoiner;

import org.osgi.framework.Bundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.http.HttpClient;
import io.vertx.core.http.HttpServerRequest;
import net.bluemind.webmodule.server.handlers.IWebModuleConsumer;
import net.bluemind.webmodule.server.js.BundleDependency;
import net.bluemind.webmodule.server.js.JsDependency;
import net.bluemind.webmodule.server.js.JsEntry;
import net.bluemind.webmodule.server.resources.IResourceProvider;

public class WebModuleBuilder {

	private static final Logger logger = LoggerFactory.getLogger(WebModuleBuilder.class);

	public String root;
	public String rootFile;
	public boolean noMaintenance;
	public String index = "index.html";

	public Map<String, HandlerFactory<HttpServerRequest>> handlers = new HashMap<>();

	public List<JsEntry> js = new ArrayList<>();
	public List<String> css = new ArrayList<>();
	public List<IResourceProvider> resources = new ArrayList<>();

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();

		sb.append(String.format("%s\n", root));
		sb.append(String.format("    index: %s\n", index));

		StringJoiner sj = new StringJoiner(",", "[", "]");
		for (IResourceProvider r : resources) {
			sj.add(r.getBundleName());
		}
		sb.append(String.format("    resources: %s \n", sj.toString()));

		sj = new StringJoiner(",", "[", "]");
		for (String handlerPath : handlers.keySet()) {
			sj.add(handlerPath);
		}
		sb.append(String.format("    handlers: %s \n", sj.toString()));

		sj = new StringJoiner(",", "[", "]");
		for (JsEntry jsPath : js) {
			sj.add(jsPath.path);
		}
		sb.append(String.format("    js: %s \n", sj.toString()));

		sj = new StringJoiner(",", "[", "]");
		for (String cssPath : css) {
			sj.add(cssPath);
		}
		sb.append(String.format("    css: %s\n", sj.toString()));

		return sb.toString();

	}

	public WebModule build(Vertx vertx, HttpClient httpClient) {
		WebModule ret = new WebModule();
		ret.root = root;
		ret.rootFile = rootFile;
		ret.noMaintenance = noMaintenance;
		ret.index = index;
		ret.handlers = new HashMap<>();
		rebuild(ret, vertx);
		for (Entry<String, HandlerFactory<HttpServerRequest>> handlerEntry : this.handlers.entrySet()) {
			Handler<HttpServerRequest> handler = handlerEntry.getValue().create(vertx);
			if (handler instanceof IWebModuleConsumer iWebModuleConsumer) {
				iWebModuleConsumer.setModule(ret);
			}

			if (handler instanceof NeedVertx needVertx) {
				needVertx.setVertx(vertx);
			}

			ret.handlers.put(handlerEntry.getKey(), handler);
		}
		return ret;
	}

	public void rebuild(WebModule module, Vertx vertx) {
		module.css.clear();
		module.css.addAll(css);
		module.js.clear();
		module.js.addAll(new OrderedJsListBuilder(js).getJsList());
		module.resources.clear();
		module.resources.addAll(resources);
	}


	private class OrderedJsListBuilder {
		private Set<String> resolved;
		private List<String> unresolved;
		private List<JsEntry> files;

		public OrderedJsListBuilder(List<JsEntry> js) {
			resolved = new LinkedHashSet<>();
			unresolved = new ArrayList<>();
			this.files = js;
		}

		public List<JsEntry> getJsList() {
			for (JsEntry file : files) {
				resolve(file);
			}
			List<JsEntry> list = new ArrayList<>();
			for (String resolvedBundle : resolved) {
				logger.info("Inject {} JS file", resolvedBundle);

				list.add(getEntryByPath(resolvedBundle));
			}

			return list;
		}

		private void resolve(JsEntry js) {
			unresolved.add(js.path);
			for (JsDependency dependency : js.getDependencies()) {
				resolveDependency(dependency, js);
			}

			if (js.getBundle() != null) {
				resolved.add(js.path);
			} else {
				logger.warn("js {} has no bundle", js.path);
			}
		}

		private void resolveDependency(JsDependency dependency, JsEntry js) {
			// should resolve every jsEntry of depBundle
			List<JsEntry> entries = dependency.getEntries(dependency, files);
			if (entries.isEmpty()) {
				throw new RuntimeException(
						"dependency " + dependency + " not found for JsEntry " + js.path + " (" + js.getBundle() + ")");
			}
			for (JsEntry entry : entries) {
				if (!resolved.contains(entry.path)) {
					if (unresolved.contains(entry.path)) {
						throw new RuntimeException("circular dependency " + js.getBundle() + " -> " + dependency);
					}
					resolve(entry);
				}
			}
		}

		private JsEntry getEntryByPath(String path) {
			for (JsEntry j : files) {
				if (j.path != null && j.path.equals(path)) {
					return j;
				}
			}
			throw new RuntimeException("dependency " + path + " not found");
		}
	}

	private Optional<IResourceProvider> findResourceBundle(String path) {
		return resources.stream().filter(resource -> resource.hasResource(path)).findFirst();
	}

	public void resolveJsBundles() {
		// resolve js
		js.stream().filter(entry -> entry.getBundle() == null).forEach(entry -> {
			findResourceBundle(entry.path).ifPresentOrElse(resourceBundle -> {
				Bundle bundle = resourceBundle.getBundle();
				entry.setBundle(bundle.getSymbolicName());
				String value = bundle.getHeaders().get("Web-Dependencies");
				if (value != null) {
					logger.debug("bundle {} depencies {}", entry.getBundle(), value);
					Arrays.asList(value.split(","))
							.forEach(dependency -> entry.addDependency(new BundleDependency(dependency)));
				}
				if (entry.hasTranslation()) {
					loadTranslations(entry, resourceBundle);
				}
			}, () -> logger.error("Could not find bundle for {}", entry.path));
		});

	}

	private void loadTranslations(JsEntry j, IResourceProvider resourceBundle) {
		String ret = j.path;
		String path = ret.substring(0, ret.lastIndexOf(".js"));
		path += "_";
		Map<String, String> translations = new HashMap<>(10);
		logger.debug("looking for {}", path);
		for (String r : resourceBundle.getResources()) {
			if (r.endsWith(".js") && r.startsWith(path)) {
				String lang = r.substring(path.length(), (path.length() - 1) + ".js".length());
				logger.debug("found translation for {} {}", lang, r);
				translations.put(lang, r);
			}
		}
		j.setTranslations(translations);
	}


}
