/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.mailbox.api.utils;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.bluemind.mailbox.api.rules.MailFilterRule;
import net.bluemind.mailbox.api.rules.conditions.MailFilterRuleCondition;
import net.bluemind.mailbox.api.rules.conditions.MailFilterRuleCondition.Operator;
import net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilter;
import net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilterContains;
import net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilterEquals;
import net.bluemind.mailbox.api.rules.conditions.MailFilterRuleFilterMatches;

public class RuleParser {

	/**
	 * visit {@link #criteria}
	 *
	 * @param handler visitor
	 */
	public static void visit(MailFilterRule rule, RuleHandler handler) {
		if (rule.conditions == null) {
			return;
		}
		rule.conditions.forEach(condition -> parseCondition(condition, handler));
	}

	/**
	 * conditionOperator {@link #criteria}
	 *
	 * @param rule MailFilterRule
	 */
	public static Operator conditionOperator(MailFilterRule rule) {
		if (rule.conditions == null || rule.conditions.isEmpty()) {
			return null;
		}

		List<Operator> ops = new ArrayList<>();
		rule.conditions.forEach(c -> parseConditionOperator(ops, c));
		return ops.get(0);
	}

	private static void parseConditionOperator(List<Operator> ops, MailFilterRuleCondition condition) {
		MailFilterRuleFilter filter = condition.filter();
		if (filter == null && !condition.conditions.isEmpty()) {
			condition.conditions.forEach(c -> parseConditionOperator(ops, c));
		}
		ops.add(condition.operator);
	}

	/**
	 * parse rule fields conditions {@link #criteria}
	 *
	 * @param rule MailFilterRule
	 */
	public static boolean conditionContainsDuplicateFields(MailFilterRule rule) {
		if (rule.conditions == null || rule.conditions.isEmpty()) {
			return false;
		}

		List<String> fields = new ArrayList<>();
		rule.conditions.forEach(c -> parseConditionFields(fields, c));
		Set<String> seen = new HashSet<>();
		return fields.stream().anyMatch(s -> !seen.add(s));
	}

	private static void parseConditionFields(List<String> fields, MailFilterRuleCondition condition) {
		MailFilterRuleFilter filter = condition.filter();
		if (filter == null && !condition.conditions.isEmpty()) {
			condition.conditions.forEach(c -> parseConditionFields(fields, c));
		}

		if (filter != null) {
			fields.addAll(condition.filter.fields);
		}
	}

	private static void parseCondition(MailFilterRuleCondition condition, RuleHandler handler) {
		MailFilterRuleFilter filter = condition.filter();
		if (filter == null) {
			if (!condition.conditions.isEmpty()) {
				condition.conditions.forEach(c -> parseCondition(c, handler));
			}
			return;
		}

		String field = convert(filter.fields.get(0));
		if (field != null) {
			handleOperation(filter, condition.negate, handler, field);
		}
	}

	private static void handleOperation(MailFilterRuleFilter filter, boolean negate, RuleHandler handler,
			String field) {
		switch (filter.operator) {
		case EXISTS:
			exists(negate, field, handler);
			break;
		case EQUALS:
			String valueEquals = firstValue(((MailFilterRuleFilterEquals) filter).values, "");
			equals(negate, field, valueEquals, handler);
			break;
		case CONTAINS:
			String valueContains = firstValue(((MailFilterRuleFilterContains) filter).values, "");
			contains(negate, field, valueContains, handler);
			break;
		case MATCHES:
			String valueMatch = firstValue(((MailFilterRuleFilterMatches) filter).values, "");
			matches(negate, field, valueMatch, handler);
			break;
		default:
			break;
		}
	}

	private static void exists(boolean negate, String field, RuleHandler handler) {
		if (!negate) {
			handler.exists(field);
		} else {
			handler.doesnotExist(field);
		}
	}

	private static void equals(boolean negate, String field, String value, RuleHandler handler) {
		if (!negate) {
			handler.is(field, value);
		} else {
			handler.isNot(field, value);
		}
	}

	private static void contains(boolean negate, String field, String value, RuleHandler handler) {
		if (!negate) {
			handler.contains(field, value);
		} else {
			handler.doesnotContain(field, value);
		}
	}

	private static void matches(boolean negate, String field, String value, RuleHandler handler) {
		if (!negate) {
			handler.matches(field, value);
		} else {
			handler.doesnotMatch(field, value);
		}
	}

	private static String firstValue(List<String> values, String orDefault) {
		return (values != null && !values.isEmpty()) ? values.get(0) : orDefault;
	}

	private static String convert(String field) {
		if (field.startsWith("from")) {
			return "FROM";
		} else if (field.startsWith("to")) {
			return "TO";
		} else if (field.startsWith("subject")) {
			return "SUBJECT";
		} else if (field.startsWith("part.content")) {
			return "BODY";
		} else if (field.startsWith("headers.")) {
			return field.split("\\.")[1];
		} else {
			return null;
		}
	}
}
