import { checkCompatibility, findCriteriaIncompatibleIndexes } from "./criteria";
import { CRITERIA_OPERATORS, CRITERIA_MATCHERS, CRITERIA_TARGETS } from "../filterRules";

describe("criteria incompatibility", () => {
    test("no criteria", () => {
        expect(checkCompatibility(criteriaBuilder().build())).toBeTruthy();
    });

    test("Two senders with AND conditions", () => {
        expect(checkCompatibility(criteriaBuilder().addTargetTypeFromMail(2).build())).toBeFalsy();
    });

    test("Two senders with OR conditions", () => {
        expect(checkCompatibility(criteriaBuilder().withOperator(CRITERIA_OPERATORS.OR).build())).toBeTruthy();
    });

    test("Two subject with EQUALS matcher and with AND conditions", () => {
        expect(
            checkCompatibility(criteriaBuilder().addTargetTypeSubject(2, CRITERIA_MATCHERS.EQUALS).build())
        ).toBeFalsy();
    });

    test("One subject with EQUALS and One Subject with CONTAINS and with AND conditions", () => {
        expect(
            checkCompatibility(
                criteriaBuilder()
                    .addTargetTypeSubject(1, CRITERIA_MATCHERS.EQUALS)
                    .addTargetTypeSubject(1, CRITERIA_MATCHERS.CONTAINS)
                    .build()
            )
        ).toBeTruthy();
    });
});

describe("find criteria incompatible", () => {
    test("no criteria", () => {
        expect(findCriteriaIncompatibleIndexes(criteriaBuilder().build())).toEqual([]);
    });

    test("Two senders with OR conditions", () => {
        expect(
            findCriteriaIncompatibleIndexes(
                criteriaBuilder().withOperator(CRITERIA_OPERATORS.OR).addTargetTypeFromMail(2).build()
            )
        ).toEqual([]);
    });

    test("Two senders with AND conditions", () => {
        expect(
            findCriteriaIncompatibleIndexes(
                criteriaBuilder().withOperator(CRITERIA_OPERATORS.AND).addTargetTypeFromMail(2).build()
            )
        ).toEqual([1]);
    });

    test("More than 2 senders with AND conditions", () => {
        expect(
            findCriteriaIncompatibleIndexes(
                criteriaBuilder().withOperator(CRITERIA_OPERATORS.AND).addTargetTypeFromMail(3).build()
            )
        ).toEqual([1, 2]);
    });
    test("More than 2 senders with AND conditions", () => {
        expect(
            findCriteriaIncompatibleIndexes(
                criteriaBuilder()
                    .withOperator(CRITERIA_OPERATORS.AND)
                    .addTargetTypeFromMail(1)
                    .addTargetToMail()
                    .addTargetTypeFromMail()
                    .build()
            )
        ).toEqual([2]);
    });
    test("More than 2 subject with AND conditions", () => {
        expect(
            findCriteriaIncompatibleIndexes(
                criteriaBuilder()
                    .withOperator(CRITERIA_OPERATORS.AND)
                    .addTargetTypeSubject()
                    .addTargetTypeFromMail(1)
                    .addTargetTypeSubject()
                    .build()
            )
        ).toEqual([2]);
    });
});

function criteriaBuilder(state) {
    let operator = state?.operator || CRITERIA_OPERATORS.AND;
    const values = state?.values || [];
    return {
        build() {
            return { operator, values };
        },
        withOperator(newOperator) {
            operator = newOperator;
            return criteriaBuilder({ operator, values });
        },
        addTargetTypeFromMail(count = 1, matcherType = CRITERIA_MATCHERS.EQUALS) {
            for (let index = 0; index < count; index++) {
                values.push({
                    isNew: false,
                    exception: false,
                    matcher: matcherType,
                    target: { type: CRITERIA_TARGETS.FROM },
                    value: {
                        address: `user${index}@devenv.dev.bluemind.net`,
                        dn: "coucou test",
                        key: index,
                        kind: "individual",
                        members: [],
                        uid: index,
                        urn: index + "@addressbook_75a0d5b3.internal"
                    }
                });
            }
            return criteriaBuilder({ operator, values });
        },
        addTargetTypeSubject(count = 1, matcherType = CRITERIA_MATCHERS.EQUALS) {
            for (let index = 0; index < count; index++) {
                values.push({
                    isNew: false,
                    exception: false,
                    matcher: matcherType,
                    target: { type: CRITERIA_TARGETS.SUBJECT },
                    value: CRITERIA_TARGETS.SUBJECT + index
                });
            }
            return criteriaBuilder({ operator, values });
        },
        addTargetToMail(count = 1, matcherType = CRITERIA_MATCHERS.EQUALS) {
            for (let index = 0; index < count; index++) {
                values.push({
                    isNew: false,
                    exception: false,
                    matcher: matcherType,
                    target: { type: CRITERIA_TARGETS.TO },
                    value: {
                        address: `user${index}@devenv.dev.bluemind.net`,
                        dn: "coucou test",
                        key: index,
                        kind: "individual",
                        members: [],
                        uid: index,
                        urn: index + "@addressbook_75a0d5b3.internal"
                    }
                });
            }
            return criteriaBuilder({ operator, values });
        }
    };
}
