import type { DeprecationTypes } from "@vue/runtime-core";
import { type Component } from "vue";

type CompatConfig = Partial<Record<DeprecationTypes, boolean | "suppress-warning">> & {
    MODE?: 2 | 3 | ((comp: Component | null) => 2 | 3);
};

const NO_WARNING = new Set([
    "Portal",
    "Panel",
    "Button",
    "SelectButton",
    "RadioButton",
    "Skeleton",
    "Tree",
    "TreeNode",
    "TreeSlots",
    "Checkbox",
    "InputGroupAddon",
    "InputText",
    "InputGroup",
    "InputNumber",
    "Chip",
    "Dialog",
    "ConfirmDialog",
    "Select",
    "Splitter",
    "SplitterPanel",
    "Popover",
    "Column",
    "DataTable",
    "TableHeader",
    "TableBody",
    "TableFooter",
    "HeaderCell",
    "BodyCell",
    "BodyRow",
    "VirtualScroller",
    "Paginator",
    "CurrentPageReport",
    "PrevPageLink",
    "NextPageLink",
    "Breadcrumb",
    "DatePicker",
    "Menu",
    "Menubar",
    "TieredMenu",
    "ProgressBar",
    "ToggleButton",
    "ChevronDownIcon"
]);

const NOT_MIGRATED = new Set(["SMimeRootRenderlessStore", "SMimeMailRenderlessStore", "RenderlessStore"]);

const configuration: CompatConfig = {
    MODE: (instance: Component | null) => {
        if (!instance || typeof instance === "function") {
            return 2;
        }
        // Component Without Name
        if (!instance.name) {
            return 2;
        }
        const name = instance.name || "";
        // vue-multipane
        if (name === "multipane") {
            return 2;
        }
        // Component that's need to be migrated.
        if (NOT_MIGRATED.has(name)) {
            console.warn(
                "This component is fallbacked to mode 2. You must make it compatible withe Vue 3 And set components compatConfig attribute. This component is not in developpement mode.",
                instance.name,
                instance
            );
            return 2;
        }
        // Bootstrap-vue
        if (/^B[A-Z]/.test(name)) {
            return 2;
        }
        // UI component and Bluetiful
        if (/^B[mv][A-Z]/.test(name)) {
            return 2;
        }
        // Component in developpement (should have a compatConfig)
        if (instance.__file) {
            console.warn(
                "This component is fallbacked to mode 2. You must make it compatible withe Vue 3 And set components compatConfig attribute.",
                instance.name,
                instance
            );
            return 2;
        }
        //MODE 3 =>
        // Build-in components
        if (instance.__isBuiltIn) {
            return 3;
        }
        // VueI18n library
        if (/^i18n/.test(name)) {
            return 3;
        }
        // Bluetiful icon library
        if (typeof instance?.template === "string" && /^<svg viewBox=/.test(instance?.template)) {
            return 3;
        }
        // GlobalEvent library
        if (name === "GlobalEvents") {
            return 3;
        }

        // Prime Vue only ? What about SMime plugin, FH plugin ?
        if (!NO_WARNING.has(name)) {
            console.warn(
                "This component is fallbacked to mode 3. If this is not right please edit compatConfiguration.ts",
                instance.name,
                instance
            );
        }
        return 3;
    }
};

export default configuration;
