package net.bluemind.webmodule.project.vite.internal;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.stream.Stream;

import org.osgi.framework.Bundle;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.json.JsonObject;
import net.bluemind.webmodule.server.js.JsEntry;
import net.bluemind.webmodule.server.js.JsEntry.Type;

public class ViteDevResource extends AbstractViteResource {
	private static final Logger logger = LoggerFactory.getLogger(ViteDevResource.class);

	private final static String DEFAULT_DEV_SERVER = "http://localhost:5173/";
	private URI server;

	public ViteDevResource(Bundle bundle, URI server) {
		super(bundle);
		logger.info("Server URL provided by configuration for {} : {}", getBundleName(), server);
		this.server = server;
	}

	public ViteDevResource(Bundle bundle) {
		super(bundle);
		logger.info("Server URL calculation set to auto for {}", getBundleName());
		server = getServerDSN();
	}

	public URI getServerDSN() {
		try {
			Path file = getFileInBundle("server.json").get();
			JsonObject config = new JsonObject(Files.readString(file));
			String host;
			if (config.getValue("host") instanceof String) {
				host = config.getString("host", "localhost");
			} else {
				host = "localhost";
			}
			Integer port = Optional.ofNullable(config.getInteger("port")).orElse(5173);
			String protocol = config.containsKey("https") ? "https" : "http";
			URI uri = new URI(protocol, null, host, port, "/", null, null);
			logger.info("Vite server configuration founded. Server URL set to {}", uri.toString());
			return uri;
		} catch (IOException | NoSuchElementException | URISyntaxException e) {
			logger.error("Could not find server.json for {}, fallback to {}", getBundleName(), DEFAULT_DEV_SERVER, e);
			return null;
		}

	}

	@Override
	public List<JsEntry> getJsEntries() {
		return Stream
				.concat(Stream.of(new JsEntry(getURL("@vite/client"), Type.MODULE, dependencies, getBundleName())),
						chunks.stream().filter(chunk -> chunk.isEntry).map(
								chunk -> new JsEntry(getURL(chunk.key), Type.MODULE, dependencies, getBundleName())))
				.toList();
	}

	private String getURL(String resource) {
		return server.resolve(resource).toString();
	}

	@Override
	public List<String> getCss() {
		return Collections.emptyList();
	}

	@Override
	protected void loadResources() {
		// No ressource needed in dev mode
	}

}
