/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.utils;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.StringTokenizer;

import com.google.common.base.Strings;

import net.bluemind.domain.api.DomainSettingsKeys;
import net.bluemind.hornetq.client.MQ.SharedMap;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.system.api.SysConfKeys;

public class KeycloakUrls {
	private static final String GLOBAL_VIRT = "global.virt";
	private static final String HTTPS = "https://";
	private static final String NO_REDIRECT_URI = HTTPS + "configure_external_url_in_bluemind";

	private final String domainUid;

	public KeycloakUrls(String domainUid) {
		this.domainUid = domainUid;
	}

	public String getExternalUrl() {
		String externalUrl = NO_REDIRECT_URI;

		if (GLOBAL_VIRT.equals(domainUid)) {
			SharedMap<String, String> sysconf = Shared.mapSysconf();
			String globalExternalUrl = sysconf.get(SysConfKeys.external_url.name());
			if (!Strings.isNullOrEmpty(globalExternalUrl)) {
				externalUrl = HTTPS + globalExternalUrl;
			}
		} else {
			Map<String, String> domainSettings = Shared.mapDomainSettings().get(domainUid);

			String domainExternalUrl;
			if (domainSettings != null
					&& (domainExternalUrl = domainSettings.get(DomainSettingsKeys.external_url.name())) != null
					&& !domainExternalUrl.isEmpty()) {
				externalUrl = HTTPS + domainExternalUrl;
			}
		}

		return externalUrl;
	}

	public Set<String> getDomainUrls() {
		Set<String> res = new HashSet<>();

		if (GLOBAL_VIRT.equals(domainUid)) {
			SharedMap<String, String> sysconf = Shared.mapSysconf();
			if (sysconf.get(SysConfKeys.external_url.name()) != null) {
				res.add(getOpenIdUrl(sysconf.get(SysConfKeys.external_url.name())));
			}

			String otherUrls = sysconf.get(SysConfKeys.other_urls.name());
			addOtherUrls(res, otherUrls);
			if (res.isEmpty()) {
				res.add(NO_REDIRECT_URI);
			}
			return res;
		}

		Map<String, String> domainSettings = Shared.mapDomainSettings().get(domainUid);
		if (domainSettings != null) {
			String domainExternalUrl;
			if ((domainExternalUrl = domainSettings.get(DomainSettingsKeys.external_url.name())) != null
					&& !domainExternalUrl.isEmpty()) {
				res.add(getOpenIdUrl(domainSettings.get(DomainSettingsKeys.external_url.name())));
			}

			String otherUrls = domainSettings.get(DomainSettingsKeys.other_urls.name());
			addOtherUrls(res, otherUrls);
			if (res.isEmpty()) {
				res.add(NO_REDIRECT_URI);
			}
			return res;
		}

		res.add(NO_REDIRECT_URI);
		return res;
	}

	private void addOtherUrls(Set<String> res, String otherUrls) {
		if (otherUrls != null) {
			StringTokenizer tokenizer = new StringTokenizer(otherUrls.trim(), " ");
			while (tokenizer.hasMoreElements()) {
				res.add(getOpenIdUrl(tokenizer.nextToken()));
			}
		}
	}

	private String getOpenIdUrl(String url) {
		return HTTPS + url + "/auth/openid";

	}
}
