/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.hornetq.client.impl;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ForkJoinPool;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.hornetq.client.MQ.IMQConnectHandler;
import net.bluemind.hornetq.client.MQKeyDB;
import net.bluemind.keydb.common.KeydbBootstrapNetAddress;

public class KeyDBClient {
	protected final Logger logger = LoggerFactory.getLogger(getClass());
	public final CompletableFuture<MQKeyDB> keyDBStart = new CompletableFuture<>();

	public KeyDBClient() {

		String keyDbIp = KeydbBootstrapNetAddress.getKeydbIP();
		logger.info("Connecting to {}....", keyDbIp);
		ForkJoinPool forComp = ForkJoinPool.commonPool();
		Thread.ofPlatform().name("bm-keydb-connect").start(() -> {
			MQKeyDB mq = null;
			do {
				try {
					mq = new MQKeyDB(new RedisConnection(keyDbIp));
					final MQKeyDB obtained = mq;
					forComp.submit(() -> keyDBStart.complete(obtained));
					logger.info("Keydb connection setup completed ({})", keyDbIp);
				} catch (Exception e) {
					logger.error(e.getMessage(), e);
					try {
						Thread.sleep(Duration.ofSeconds(5));
					} catch (InterruptedException ie) {
						Thread.currentThread().interrupt();
						forComp.submit(() -> keyDBStart.completeExceptionally(ie));
						break;
					}
				}
			} while (mq == null);
		});
	}

	public CompletableFuture<Void> init() {
		CompletableFuture<Void> initFuture = new CompletableFuture<>();
		keyDBStart.whenComplete((keydb, ex) -> {
			if (ex == null) {
				initFuture.complete(null);
			} else {
				initFuture.completeExceptionally(ex);
			}
		});
		return initFuture;
	}

	public final synchronized void init(final IMQConnectHandler handler) {
		keyDBStart.whenComplete((keydb, ex) -> {
			if (ex == null) {
				handler.connected();
			}
		});
	}
}
