package net.bluemind.cql.node.hook;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.cql.CqlKeyspaceService;
import net.bluemind.cql.CqlSchemaService;
import net.bluemind.cql.CqlSessions;
import net.bluemind.cql.node.api.ICqlNode;
import net.bluemind.cql.persistence.CqlKeyspaceStore;
import net.bluemind.cql.persistence.CqlSchemaStore;
import net.bluemind.network.utils.NetworkHelper;
import net.bluemind.server.api.IServer;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.server.hook.DefaultServerHook;

public class CqlNodeHook extends DefaultServerHook {

	private static final Logger logger = LoggerFactory.getLogger(CqlNodeHook.class);

	@Override
	public void onServerTagged(BmContext context, ItemValue<Server> itemValue, String tag) throws ServerFault {
		if (tag.equals(TagDescriptor.cql_node.getTag())) {
			ICqlNode cqlNodeApi = context.provider().instance(ICqlNode.class);
			cqlNodeApi.addNode(itemValue);
			IServer srvApi = context.provider().instance(IServer.class, "default");
			long otherServers = srvApi.allComplete().stream()
					.filter(s -> s.value.tags.contains(TagDescriptor.cql_node.getTag()))
					.filter(s -> !s.uid.equals(itemValue.uid)).count();
			if (otherServers == 0) {
				logger.info("Init schema for first CQL node {}...", itemValue.value);
				long ms = CoreConfig.get().getDuration(CoreConfig.Cassandra.PORT_TIMEOUT, TimeUnit.MILLISECONDS);
				new NetworkHelper(itemValue.value.address()).waitForListeningPort(9042, ms, TimeUnit.MILLISECONDS);
				initSchema(context.provider().instance(IServer.class, "default"));
			}
		}
	}

	private void initSchema(IServer api) {
		try (CqlSession systemSession = CqlSessions.system(api)) {
			CqlKeyspaceStore store = new CqlKeyspaceStore(systemSession);
			CqlKeyspaceService service = new CqlKeyspaceService(store);
			List<String> spaces = service.initialize();
			for (String ks : spaces) {
				CqlSchemaStore ss = new CqlSchemaStore(CqlSessions.forKeyspace(ks, api));
				CqlSchemaService schemas = new CqlSchemaService(ks, ss);
				schemas.initialize();
			}
		}

	}

	@Override
	public void onServerUntagged(BmContext context, ItemValue<Server> itemValue, String tag) throws ServerFault {
		if (tag.equals(TagDescriptor.cql_node.getTag())) {
			ICqlNode cqlNodeApi = context.provider().instance(ICqlNode.class);
			cqlNodeApi.removeNode(itemValue.uid);
		}
	}

}
