package net.bluemind.pimp;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import org.eclipse.equinox.app.IApplication;
import org.eclipse.equinox.app.IApplicationContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.ByteStreams;
import com.google.common.io.Files;

import net.bluemind.pimp.impl.Rule;
import net.bluemind.pimp.impl.RulesBuilder;
import net.bluemind.pimp.impl.UsageFile;

public class PimpMyRam implements IApplication {
	private static final Logger logger = LoggerFactory.getLogger(PimpMyRam.class);

	@Override
	public Object start(IApplicationContext context) throws Exception {
		Thread.setDefaultUncaughtExceptionHandler(
				(thread, exception) -> logger.error("Unhandled exception in {}", thread, exception));

		pimpSysCtl();
		pimpSysfs();

		Rule[] rules = loadRules();
		printMemoryAllocation(rules);
		long totalMemMB = getTotalSystemMemory();
		int spareMb = configureSpareMemory(rules, totalMemMB);
		configureProductMemory(rules, spareMb);
		if (hasPostgresql()) {
			pimpPostgresql(totalMemMB);
		}

		System.exit(0);
		return IApplication.EXIT_OK;
	}

	private void pimpSysCtl() {
		try (InputStream in = PimpMyRam.class.getClassLoader().getResourceAsStream("data/sysctl/bm.conf")) {
			Files.write(ByteStreams.toByteArray(in), new File("/etc/sysctl.d/01-bluemind.conf"));

			int ret = SystemHelper.cmd("sysctl", "--system");
			if (ret != 0) {
				logger.warn("Loading sysctl ending with error code {}", ret);
			}
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
		}
	}

	private void pimpSysfs() {
		try (InputStream in = PimpMyRam.class.getClassLoader().getResourceAsStream("data/sysfs/bm.conf")) {
			Files.write(ByteStreams.toByteArray(in), new File("/etc/tmpfiles.d/01-bluemind-sysfs.conf"));

			int ret = SystemHelper.cmd("systemd-tmpfiles", "--create");
			if (ret != 0) {
				logger.warn("systemd-tmpfiles --create ends with error code {}", ret);
			}
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
		}
	}

	public boolean hasPostgresql() {
		return new File("/usr/share/doc/bm-postgresql/").isDirectory();
	}

	private void pimpPostgresql(long totalMemMB) {
		// this would be better if we include a file in the package
		boolean isShard = new File("/usr/share/doc/bm-mailbox-role/").isDirectory();
		if (totalMemMB > 63000) {
			writePg(isShard ? "mem.shard.64g" : "mem.64g");
		} else if (totalMemMB > 47000) {
			writePg(isShard ? "mem.shard.48g" : "mem.48g");
		} else if (totalMemMB > 31000) {
			writePg(isShard ? "mem.shard.32g" : "mem.32g");
		} else if (totalMemMB > 15000) {
			writePg(isShard ? "mem.shard.16g" : "mem.16g");
		} else {
			writePg("mem.default");
		}
	}

	private void writePg(String tplName) {
		try (InputStream in = PimpMyRam.class.getClassLoader().getResourceAsStream("data/pg/" + tplName);
				ByteArrayOutputStream bos = new ByteArrayOutputStream()) {
			bos.write(
					"# DO NOT MODIFY\n# OVERWRITTEN BY bm-pimp\n# use postgresql.conf.local for specific configuration\n"
							.getBytes());
			bos.write(ByteStreams.toByteArray(in));

			Files.write(bos.toByteArray(), new File("/etc/postgresql/17/main/postgresql.conf.pimp"));
			logger.info("PostgreSQL memory configured ({})", tplName);
		} catch (IOException e) {
			logger.error(e.getMessage(), e);
		}
	}

	private Rule[] loadRules() {
		return new RulesBuilder().build();
	}

	private void configureProductMemory(Rule[] rules, int spareMb) throws IOException {
		File parent = new File("/etc/bm/default");
		parent.mkdirs();
		int sparePercentAlloc = 0;

		UsageFile pred = UsageFile.of("/etc/bm/usage-prediction.json");
		UsageFile metric = UsageFile.of("/etc/bm/usage-from-metrics.json");

		for (Rule r : rules) {
			if (!productEnabled(r.getProduct())) {
				logger.info("{} {}is not installed or disabled, not configuring.", r.getProduct(),
						r.isOptional() ? "(optional) " : "");
				continue;
			}
			sparePercentAlloc += r.getSparePercent();

			File f = confPath(r);
			int fromSpare = spareMb / 100 * r.getSparePercent();
			// our stack size is at 256k so we divide by 4 to get MB
			int cpuBoostMb = r.getCpusBoost() * Runtime.getRuntime().availableProcessors() / 4;
			if (cpuBoostMb > 0) {
				logger.info("CPU boost is {}MB", cpuBoostMb);
			}

			int usageMb = 0;
			if (r.getUsageVar() != null) {
				int predicted = pred.usage.computeIfAbsent(r.getUsageVar(), k -> 0);
				int measured = metric.usage.computeIfAbsent(r.getUsageVar(), k -> 0);
				int considered = Math.max(predicted, measured);
				usageMb = considered * r.getUsageBonusKB() / 1024;
				logger.info("Usage bonus for {} -> {}MB", r.getProduct(), usageMb);
			}

			int memMb = r.getDefaultHeap() + fromSpare + cpuBoostMb + usageMb;

			int dmemMb = Math.min(r.getDirectCap(), r.getDefaultDirect() + fromSpare);
			String content = "MEM=" + memMb + "\nDMEM=" + dmemMb + "\n";

			logger.info("  * {} gets +{}MB for a total of {}MB", r.getProduct(), fromSpare, memMb);
			Files.write(content.getBytes(), f);

			// also write to the old location
			File oldDir = new File("/etc/" + r.getProduct() + "/");
			if (oldDir.mkdirs()) {
				File oldFile = new File(oldDir, "mem_conf.ini");
				if (!oldFile.exists()) {
					Files.write(content.getBytes(), oldFile);
				}
			}
		}
		logger.info("Spare percent allocation is set at {}% in rules.json", sparePercentAlloc);
	}

	private int configureSpareMemory(Rule[] rules, long totalMemMB) {
		int spareMB = (int) totalMemMB;
		logger.info("Initial free memory : {}MB", spareMB);

		// Postgres is enabled
		if (new File("/usr/lib/systemd/system/bm-postgresql.service").exists()) {
			spareMB -= 6144;
			logger.info("Postgresql is enabled on this node, decreasing free memory to {}MB (-{}MB).", spareMB, 6144);
		}

		for (Rule r : rules) {
			if (productEnabled(r.getProduct())) {
				spareMB -= r.getDefaultHeap();
				logger.info("{} is enabled on this node, decreasing free memory to {}MB (-{}MB)", r.getProduct(),
						spareMB, r.getDefaultHeap());
			}
		}

		if (spareMB < 0) {
			logger.warn("No spare memory to distribute to JVMs ({})", spareMB);
			System.exit(0);
		}

		spareMB *= 0.40;
		logger.info("Taking {}% of free memory as spare to allocate to JVMs : {}MB", 0.40 * 100, spareMB);

		return spareMB;

	}

	private long getTotalSystemMemory()
			throws NoSuchMethodException, IllegalAccessException, InvocationTargetException {
		OperatingSystemMXBean osMxBean = ManagementFactory.getOperatingSystemMXBean();
		Method totalMem = osMxBean.getClass().getMethod("getTotalPhysicalMemorySize");
		long totalMemMB = 4096;
		if (totalMem != null) {
			totalMem.setAccessible(true);
			totalMemMB = ((Long) totalMem.invoke(osMxBean)) / 1024 / 1024;
			logger.info("Total from JMX: {}MB", totalMemMB);
		} else {
			logger.error("Cannot figure out physical memory size");
			System.exit(1);
		}
		return totalMemMB;
	}

	private void printMemoryAllocation(Rule[] rules) {
		int totalPercent = 0;
		int totalDefaultMb = 0;
		for (Rule r : rules) {
			if (productEnabled(r.getProduct())) {
				totalPercent += r.getSparePercent();
				totalDefaultMb += r.getDefaultHeap();
			}
		}
		logger.info("{}MB is allocated for all heaps.", totalDefaultMb);
		validateTotalMemoryPercentage(totalPercent);
		logger.info("{}% of spare memory will be allocated to java components", totalPercent);
	}

	private void validateTotalMemoryPercentage(int totalPercent) {
		if (totalPercent > 100) {
			logger.error("You cannot distribute more than 100% of spare memory, total is {}%", totalPercent);
			System.exit(1);
		}
	}

	private File confPath(Rule r) {
		return new File("/etc/bm/default/" + r.getProduct() + ".ini");
	}

	@Override
	public void stop() {
		// ok
	}

	private boolean productEnabled(String productName) {
		File productDir = new File("/usr/share/" + productName);
		return productDir.exists() && !new File("/etc/bm/" + productName + ".disabled").exists();
	}
}
