/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.backup.continuous.events.bodies;

import java.io.File;
import java.io.InputStream;
import java.nio.file.Files;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.LongAdder;

import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.common.serialization.Serde;
import org.apache.kafka.common.serialization.Serdes;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.streams.KafkaStreams;
import org.apache.kafka.streams.KafkaStreams.State;
import org.apache.kafka.streams.KafkaStreams.StateListener;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.StreamsBuilder;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.Topology;
import org.apache.kafka.streams.errors.StreamsUncaughtExceptionHandler.StreamThreadExceptionResponse;
import org.apache.kafka.streams.kstream.Produced;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Verticle;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.json.JsonObject;
import net.bluemind.config.InstallationId;
import net.bluemind.core.backup.continuous.DefaultBackupStore;
import net.bluemind.core.backup.continuous.api.IBackupStore;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.domain.api.IDomains;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;

public class BodiesMigrationVerticle extends AbstractVerticle {

	private static final Logger logger = LoggerFactory.getLogger(BodiesMigrationVerticle.class);
	private LongAdder bodies = new LongAdder();

	@Override
	public void start() throws Exception {
		VertxPlatform.executeBlockingTimer(vertx, 10000, this::startImpl);
	}

	private void startImpl(long unused) {
		if (StateContext.getState() != SystemState.CORE_STATE_RUNNING) {
			VertxPlatform.executeBlockingTimer(vertx, 10000, this::startImpl);
			return;
		}
		if (Boolean.getBoolean(SysConfKeys.kafka_bodies_migrated.name())) {
			VertxPlatform.executeBlockingTimer(vertx, 10000, this::startImpl);
			return;
		}

		Bootstrap servers = kafkaBootstrapServers();
		if (!servers.valid()) {
			VertxPlatform.executeBlockingTimer(vertx, 30000, this::startImpl);
			return;
		}
		ServerSideServiceProvider prov = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);

		ISystemConfiguration confApi = prov.instance(ISystemConfiguration.class);
		SystemConf values = confApi.getValues();
		Boolean migrationDone = values.booleanValue(SysConfKeys.kafka_bodies_migrated.name(), false);
		if (migrationDone != null && migrationDone.booleanValue()) {
			logger.info("Bodies migration is complete, no need to start over.");
			return;
		}

		String prefix = InstallationId.getIdentifier().replace("bluemind-", "").replace("-", "");

		IDomains domApi = prov.instance(IDomains.class);
		List<String> domainTopics = domApi.all().stream().filter(iv -> !iv.value.global)
				.map(iv -> prefix + "-" + iv.uid).toList();

		String outputTopic = prefix + "-bodies.store";
		logger.info("Steam to {}", outputTopic);

		withValidClassloader(() -> initStream(servers, domainTopics, outputTopic));

	}

	private void initStream(Bootstrap servers, List<String> domainTopics, String outputTopic) {
		JsonObject empty = JsonObject.of();
		KeyValue<JsonObject, JsonObject> drop = new KeyValue<>(empty, empty);

		BaseContainerDescriptor bcd = BaseContainerDescriptor.create("osef", "name", "aa_owner", "message_bodies",
				"bodies.store", true);
		IBackupStore<Object> store = DefaultBackupStore.store().forContainer(bcd);
		logger.info("Created {} to ensure topic exists and is setup properly", store);

		Properties props = new Properties();
		props.put(StreamsConfig.APPLICATION_ID_CONFIG, "core-body-migration-" + InstallationId.getIdentifier());
		props.put(StreamsConfig.BOOTSTRAP_SERVERS_CONFIG, servers.brokers());
		props.put(StreamsConfig.NUM_STREAM_THREADS_CONFIG, Math.min(Runtime.getRuntime().availableProcessors(), 4));
		props.put(StreamsConfig.DEFAULT_KEY_SERDE_CLASS_CONFIG, Serdes.ByteArraySerde.class.getName());
		props.put(StreamsConfig.DEFAULT_VALUE_SERDE_CLASS_CONFIG, Serdes.ByteArraySerde.class.getName());
		props.put(StreamsConfig.producerPrefix(ProducerConfig.MAX_REQUEST_SIZE_CONFIG), 12 * 1024 * 1024);

		StreamsBuilder builder = new StreamsBuilder();
		builder//
				.<byte[], byte[]>stream(domainTopics)//
				.map((byte[] key, byte[] value) -> {
					JsonObject k = new JsonObject(Buffer.buffer(key));
					String type = k.getString("type");
					switch (type) {
					case "message_bodies":
						bodies.increment();
						return bodyKeyValue(k, value);
					case "message_bodies_es_source":
						bodies.increment();
						return esSourceKeyValue(k, value);
					default:
						return drop;
					}

				})//
				.filter((k, v) -> k != empty)//
				.to(outputTopic, producer());

		createStream(builder.build(), props);

	}

	private void withValidClassloader(Runnable r) {
		ClassLoader savedCl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(null);
		try {
			r.run();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
		} finally {
			Thread.currentThread().setContextClassLoader(savedCl);
		}
	}

	private void createStream(Topology topology, Properties props) {
		KafkaStreams stream = new KafkaStreams(topology, props);
		stream.setUncaughtExceptionHandler(t -> {
			logger.error(t.getMessage(), t);
			return StreamThreadExceptionResponse.SHUTDOWN_CLIENT;
		});
		stream.setStateListener(new MigStateListener(stream, vertx, bodies));
		stream.start();
	}

	private KeyValue<JsonObject, JsonObject> esSourceKeyValue(JsonObject k, byte[] value) {
		JsonObject body = new JsonObject(Buffer.buffer(value));
		String guid = body.getString("uid");
		String keyPrefix = guid.substring(0, 2);
		k.put("owner", keyPrefix + "_owner");
		k.put("uid", keyPrefix + "_owner_at_bodies.store_message_bodies");
		k.put("operation", "CREATE");
		return new KeyValue<>(k, body);
	}

	private KeyValue<JsonObject, JsonObject> bodyKeyValue(JsonObject k, byte[] value) {
		JsonObject body = new JsonObject(Buffer.buffer(value));
		String guid = body.getJsonObject("value").getString("guid");
		String keyPrefix = guid.substring(0, 2);
		k.put("owner", keyPrefix + "_owner");
		k.put("uid", keyPrefix + "_owner_at_bodies.store_message_bodies_es_source");
		k.put("operation", "CREATE");
		return new KeyValue<>(k, body);
	}

	Produced<JsonObject, JsonObject> producer() {
		Serde<JsonObject> ser = serdes();
		return Produced.<JsonObject, JsonObject>with(ser, ser, (String topic, JsonObject key, JsonObject value,
				int numPartitions) -> Optional.of(Set.of(Math.abs(key.getString("owner").hashCode() % numPartitions))));

	}

	private Serde<JsonObject> serdes() {
		return new Serde<JsonObject>() {

			@Override
			public Serializer<JsonObject> serializer() {
				return (String topic, JsonObject data) -> data.encode().getBytes();
			}

			@Override
			public Deserializer<JsonObject> deserializer() {
				return new Deserializer<JsonObject>() {

					@Override
					public JsonObject deserialize(String topic, byte[] data) {
						return new JsonObject(Buffer.buffer(data));
					}
				};
			}
		};
	}

	private static record Bootstrap(String zookeeper, String brokers) {
		boolean valid() {
			return zookeeper != null && brokers != null;
		}
	}

	private Bootstrap kafkaBootstrapServers() {
		String brokersBootstrap = System.getProperty("bm.kafka.bootstrap.servers");
		String zkBootstrap = System.getProperty("bm.zk.servers");
		if (brokersBootstrap == null || zkBootstrap == null) {
			File local = new File("/etc/bm/kafka.properties");
			if (!local.exists()) {
				local = new File(System.getProperty("user.home") + "/kafka.properties");
			}
			if (local.exists()) {
				Properties tmp = new Properties();
				try (InputStream in = Files.newInputStream(local.toPath())) {
					tmp.load(in);
				} catch (Exception e) {
					logger.warn(e.getMessage());
				}
				brokersBootstrap = tmp.getProperty("bootstrap.servers");
				zkBootstrap = tmp.getProperty("zookeeper.servers");
			}
		}
		return new Bootstrap(zkBootstrap, brokersBootstrap);
	}

	private static class MigStateListener implements StateListener {

		private long periodic = -1;
		private Vertx vertx;
		private KafkaStreams stream;
		private LongAdder processed;

		public MigStateListener(KafkaStreams stream, Vertx vertx, LongAdder processed) {
			this.stream = stream;
			this.vertx = vertx;
			this.processed = processed;

		}

		@Override
		public void onChange(State newState, State oldState) {
			logger.info("state {} -> {}", oldState, newState);
			if (newState == State.RUNNING) {
				periodic = vertx.setPeriodic(60000, tid -> {
					ConcurrentHashMap<String, Long> lags = new ConcurrentHashMap<>();
					stream.metadataForLocalThreads().forEach(tm -> {
						for (var at : tm.activeTasks()) {
							var commits = at.committedOffsets();
							var ends = at.endOffsets();
							commits.forEach((tp, commit) -> {
								long endOfset = ends.getOrDefault(tp, commit);
								long lag = endOfset - commit;
								lag = lag < 0 ? 0 : lag;
								String k = at.taskId() + "-" + tp.topic() + "#" + tp.partition();
								lags.put(k, lag);
							});
						}
					});
					long globalLag = lags.reduceValuesToLong(8, Long::longValue, 0L, (l1, l2) -> l1 + l2);
					logger.info("lag is {}, processed {}", globalLag, processed.sum());
					if (globalLag == 0) {
						vertx.executeBlocking(() -> {
							ServerSideServiceProvider prov = ServerSideServiceProvider
									.getProvider(SecurityContext.SYSTEM);
							ISystemConfiguration confApi = prov.instance(ISystemConfiguration.class);
							confApi.updateMutableValues(Map.of(SysConfKeys.kafka_bodies_migrated.name(), "true"));
							logger.info("Closing kstream as we moved everything");
							stream.close();
							return null;
						});
					}
				});
			} else {
				vertx.cancelTimer(periodic);
			}
		}

	}

	public static class Factory implements IVerticleFactory, IUniqueVerticleFactory {

		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new BodiesMigrationVerticle();
		}

	}

}
