/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2021
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.backup.continuous.events;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Promise;
import io.vertx.core.Verticle;
import io.vertx.core.eventbus.MessageConsumer;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.backup.continuous.dto.ContainerItemIdSeq;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.repository.sequences.IItemSequencesStore;
import net.bluemind.repository.sequences.IItemSequencesStore.Seq;
import net.bluemind.system.api.SystemState;

public class ContainerItemIdSeqVerticle extends AbstractVerticle {
	private static final Logger logger = LoggerFactory.getLogger(ContainerItemIdSeqVerticle.class);
	private static final String TYPE = "container_item_id_seq";

	public static class Factory implements IVerticleFactory, IUniqueVerticleFactory {

		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new ContainerItemIdSeqVerticle();
		}

	}

	@Override
	public void start(Promise<Void> startPromise) throws Exception {
		MessageConsumer<JsonObject> stateConsumer = vertx.eventBus().consumer(SystemState.BROADCAST);
		stateConsumer.handler(m -> {
			SystemState state = SystemState.fromOperation(m.body().getString("operation"));
			if (state == SystemState.CORE_STATE_RUNNING) {
				stateConsumer.unregister();
				VertxPlatform.executeBlockingPeriodic(vertx, TimeUnit.MINUTES.toMillis(1), timerId -> {
					ContainerItemIdSeq itemIdSeq = fetchContainerItemIdSeq();
					backupContainerItemIdSeq(itemIdSeq);
				});
			}
		});
		startPromise.complete();
	}

	private ContainerItemIdSeq fetchContainerItemIdSeq() {
		ContainerItemIdSeq maxItemIds = new ContainerItemIdSeq();
		IItemSequencesStore seqApi = RepositoryProvider.instance(IItemSequencesStore.class, rootCtx());
		List<Seq> seqs = seqApi.activeSequences();
		maxItemIds.defaultDataSourceSeq = seqs.stream().filter(s -> s.name().equals("directory")).findAny()
				.map(Seq::curval).orElse(0L);
		seqs.stream().filter(s -> !s.name().equals("directory"))
				.forEach(s -> maxItemIds.mailboxDataSourceSeq.put(s.name(), s.curval()));

		return maxItemIds;
	}

	private BmContext rootCtx() {
		return ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();
	}

	private void backupContainerItemIdSeq(ContainerItemIdSeq itemIdSeq) {
		ContainerDescriptor metaDesc = ContainerDescriptor.create(TYPE, TYPE, "system", TYPE, null, true);
		ItemValue<ContainerItemIdSeq> metaItem = ItemValue.create(TYPE, itemIdSeq);
		metaItem.internalId = metaItem.uid.hashCode();
		TxOutboxLookup.forContainer(metaDesc).forKafka(metaItem, null, false);
	}

}
