/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.user.service.internal.cql;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ItemValueExists;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.ItemVersion;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.utils.ValidationResult;
import net.bluemind.directory.api.BaseDirEntry.AccountType;
import net.bluemind.directory.cql.common.CqlContStoreService;
import net.bluemind.directory.repository.DirEntryNG;
import net.bluemind.domain.api.Domain;
import net.bluemind.group.api.Group;
import net.bluemind.user.api.ChangePassword;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.PasswordInfo;
import net.bluemind.user.api.User;
import net.bluemind.user.hook.IUserHook;
import net.bluemind.user.persistence.security.HashFactory;
import net.bluemind.user.service.IInCoreUser;
import net.bluemind.user.service.internal.BaseUserService;

public class CqlUserService extends BaseUserService implements IInCoreUser, IUser {

	private static final UserDirAdapter adapter = new UserDirAdapter();
	private static final Logger logger = LoggerFactory.getLogger(CqlUserService.class);

	private final CqlContStoreService store;
	private final ItemValue<Domain> domain;
	private final List<IUserHook> userHooks;
	private final BmContext context;

	public CqlUserService(BmContext ctx, Container dir, ItemValue<Domain> domain, List<IUserHook> userHooks) {
		super(ctx, domain);
		this.context = ctx;
		this.store = new CqlContStoreService(ctx, dir, domain);
		this.domain = domain;
		this.userHooks = userHooks;
	}

	@Override
	public void setPhoto(String uid, byte[] photo) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public byte[] getPhoto(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void deletePhoto(String uid) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public void setExtId(String uid, String extId) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public User get(String uid) {
		return Optional.ofNullable(getComplete(uid)).map(u -> u.value).orElse(null);
	}

	@Override
	public void restore(ItemValue<User> item, boolean isCreate) {
		// TODO Auto-generated method stub

	}

	@Override
	public ItemValueExists itemValueExists(String uid) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<ItemValue<Group>> memberOf(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<String> memberOfGroups(String uid) throws ServerFault {
		Item item = ServerFault.onException(() -> store.getItemStore().get(uid), ErrorCode.SQL_ERROR);
		if (item != null) {
			ItemValue<DirEntryNG> entry = store.get(item.id, null);
			List<String> directGroups = entry.value.memberOf != null ? new ArrayList<>(entry.value.memberOf)
					: Collections.emptyList();
			for (String gUid : directGroups) {
				directGroups.addAll(memberOfGroups(gUid));
			}
			return directGroups;
		}
		return Collections.emptyList();
	}

	@Override
	public ValidationResult validate(String[] uids) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void create(String uid, User user) throws ServerFault {
		createWithExtId(uid, null, user);
	}

	@Override
	public void createWithExtId(String uid, String extId, User user) throws ServerFault {
		DirEntryNG de = adapter.fromUser(uid, domain, user);
		for (var hook : userHooks) {
			try {
				hook.beforeCreate(context, domain.uid, uid, user);
			} catch (Exception e) {
				logger.warn("Could not run {} beforeCreate", hook);
			}
		}
		ItemVersion iv = store.createWithId(uid, null, extId, de.displayName, de);
		ItemValue<User> asItemValue = ItemValue.create(Item.create(uid, iv.id), user);
		for (var hook : userHooks) {
			try {
				hook.onUserCreated(context, domain.uid, asItemValue);
			} catch (Exception e) {
				logger.warn("Could not run {} onUserCreated", hook);
			}
		}
	}

	@Override
	public void update(String uid, User user) throws ServerFault {
		DirEntryNG de = adapter.fromUser(uid, domain, user);
		store.update(uid, de.displayName, de);
	}

	@Override
	public ItemValue<User> getComplete(String uid) throws ServerFault {
		ItemValue<DirEntryNG> entry = store.get(uid, null);
		return Optional.ofNullable(entry).map(i -> ItemValue.create(i, adapter.fromEntry(i.value))).orElse(null);
	}

	@Override
	public ItemValue<User> getLight(String uid) throws ServerFault {
		return getComplete(uid);
	}

	@Override
	public ItemValue<User> getFull(String uid, boolean useCache) {
		return getComplete(uid);
	}

	@Override
	public ItemValue<User> byEmail(String email) throws ServerFault {
		Item foundItem = store.byEmail(email);
		logger.info("byEmail({}) -> {}", email, foundItem);
		return Optional.ofNullable(foundItem).map(i -> getComplete(foundItem.uid)).orElse(null);
	}

	@Override
	public ItemValue<User> byExtId(String extId) throws ServerFault {
		return ServerFault.onException(() -> {
			Item found = store.getItemStore().getByExtId(extId);
			return Optional.ofNullable(found).map(i -> getComplete(found.uid)).orElse(null);
		}, ErrorCode.SQL_ERROR);
	}

	@Override
	public ItemValue<User> byLogin(String login) throws ServerFault {
		Item foundItem = store.byMailboxName(login);
		return Optional.ofNullable(foundItem).map(i -> getComplete(foundItem.uid)).orElse(null);
	}

	@Override
	public TaskRef delete(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<String> allUids() throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<String> allUids(boolean withLoginCaps) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void setRoles(String uid, Set<String> roles) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public Set<String> getUsersWithRoles(List<String> roles) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public Set<String> getResolvedRoles(String uid) throws ServerFault {
		return directResolvedRoles(uid, memberOfGroups(uid));
	}

	@Override
	public Set<String> directResolvedRoles(String userUid, List<String> groups) throws ServerFault {
		Set<String> mut = new HashSet<>(getRoles(userUid));
		groups.forEach(gUid -> mut.addAll(getRoles(gUid)));
		System.err.println("resolveRoles(" + userUid + ", " + groups + ") -> " + mut);
		return mut;
	}

	@Override
	public Set<String> getRoles(String uid) throws ServerFault {
		Item item = ServerFault.onException(() -> store.getItemStore().get(uid), ErrorCode.SQL_ERROR);
		if (item != null) {
			return store.getRoles(item);
		}
		return Collections.emptySet();
	}

	@Override
	public void setPassword(String uid, ChangePassword password) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public byte[] getIcon(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void updateVCard(String uid, VCard userVCard) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public VCard getVCard(String uid) throws ServerFault {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public void enablePerUserLog(String userUid, String endpoint, boolean enable) {
		// TODO Auto-generated method stub

	}

	@Override
	public void updateAccountType(String uid, AccountType accountType) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public void deleteUserIdentitiesForMailbox(String uid) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public void deleteUserIdentitiesForMailbox(String userUid, String mailboxUid) throws ServerFault {
		// TODO Auto-generated method stub

	}

	@Override
	public PasswordInfo getPasswordInfo(String login, String password) {
		System.err.println("getPwInfo for '" + login + "'");
		Item it = store.byMailboxName(login);
		if (it != null) {
			ItemValue<DirEntryNG> entry = store.get(it.id, null);
			if (entry.value.loginCaps != null) {
				String hash = entry.value.loginCaps.passwordHash;
				boolean valid = HashFactory.getByPassword(hash).validate(password, hash);
				return new PasswordInfo(valid, false, entry.uid);
			}
		}
		return new PasswordInfo(false, false, null);
	}

	@Override
	public List<ItemValue<User>> getMultipleVcardOnly(List<String> uids) throws ServerFault {
		// TODO Auto-generated method stub
		return Collections.emptyList();
	}

	@Override
	public String getPassword(long userId) {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public boolean checkApiKey(String userUid, String sid) {
		return false;
	}

}
