/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.lib.srs;

import java.util.concurrent.TimeUnit;

import com.google.common.base.Strings;

public class SrsTimestamp {
	@SuppressWarnings("serial")
	private static class InvalidChar extends RuntimeException {
	}

	private static final int TIME_PRECISION = (int) TimeUnit.DAYS.toSeconds(1);

	private static final String TIME_BASE_CHARS = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";
	private static final char[] TIME_BASE_CHARS_ARRAY = TIME_BASE_CHARS.toCharArray();
	private static final int TIME_BASE_BITS = 5; // 2^5 = 32 = timeBaseChars.lenght

	private static final int TIME_SIZE = 2; // Encoded timestamp string length
	// number of possible timestamps values
	private static final int TIME_SLOT = (int) Math.pow(TIME_BASE_CHARS.length(), TIME_SIZE);

	private static final int MAX_AGE = 10; // SRS timestamp maximum validity in days

	private SrsTimestamp() {
		// Don't use it
	}

	public static String from(long timestamp) {
		long ts = (int) timestamp / TIME_PRECISION;
		int second = (int) (ts & ((1 << TIME_BASE_BITS) - 1));

		ts = ts >> TIME_BASE_BITS;
		int first = (int) (ts & ((1 << TIME_BASE_BITS) - 1));

		return new StringBuilder().append(TIME_BASE_CHARS_ARRAY[first]).append(TIME_BASE_CHARS_ARRAY[second])
				.toString();
	}

	public static boolean check(String srsTimeStamp) {
		if (Strings.isNullOrEmpty(srsTimeStamp) || srsTimeStamp.length() > 2) {
			return false;
		}

		char[] sts = srsTimeStamp.toCharArray();

		int then = 0;
		try {
			then = updateThen(then, sts[0]);
			then = updateThen(then, sts[1]);
		} catch (InvalidChar ic) {
			return false;
		}

		long now = (System.currentTimeMillis() / TimeUnit.SECONDS.toMillis(1) / TIME_PRECISION) % TIME_SLOT;

		while (now < then) {
			now = now + TIME_SLOT;
		}

		return now <= then + MAX_AGE;
	}

	private static int updateThen(int then, char c) {
		int pos = TIME_BASE_CHARS.indexOf(Character.toUpperCase(c));
		if (pos == -1) {
			throw new InvalidChar();
		}

		return (then << TIME_BASE_BITS) | pos;
	}
}
