package net.bluemind.memory.pool.mmap;

import java.io.IOException;
import java.io.OutputStream;
import java.lang.foreign.MemorySegment;
import java.nio.ByteBuffer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.memory.pool.api.IWritableChunk;

public class WritableChunk extends Chunk implements IWritableChunk {

	private static final Logger logger = LoggerFactory.getLogger(WritableChunk.class);

	private int currentPosition;

	public WritableChunk(Segment parent, int samplePeriod, long startOffset, int size) {
		super(parent, samplePeriod, startOffset, size);
		this.currentPosition = 0;
	}

	/**
	 * Ajoute des données au chunk
	 */
	public void append(byte[] bytes) {
		append(bytes, 0, bytes.length);
	}

	@Override
	public int getSize() {
		return getPosition();
	}

	/**
	 * Ajoute des données au chunk
	 */
	public void append(byte[] bytes, int offset, int length) {
		if (released.get()) {
			throw new IllegalStateException("Chunk is released");
		}
		if (currentPosition + length > size) {
			throw new IllegalArgumentException("Not enough space in chunk");
		}

		copy(MemorySegment.ofArray(bytes), offset, length);
	}

	private void copy(MemorySegment src, int offset, int length) {
		MemorySegment segment = parent.getMemorySegment();
		try {
			MemorySegment.copy(src, offset, segment, startOffset + currentPosition, length);
		} catch (InternalError ie) {
			logger.error("Error writing to {}, start: {}, currentPos: {}, length: {}", segment, startOffset,
					currentPosition, length);
			throw ie;
		}
		currentPosition += length;
	}

	/**
	 * Ajoute des données depuis un ByteBuffer
	 */
	public void append(ByteBuffer buffer) {
		if (released.get()) {
			throw new IllegalStateException("Chunk is released");
		}
		int length = buffer.remaining();
		if (currentPosition + length > size) {
			throw new IllegalArgumentException("Not enough space in chunk");
		}

		copy(MemorySegment.ofBuffer(buffer), 0, length);
	}

	/**
	 * Positionne le curseur d'écriture
	 */
	public void seek(int pos) {
		if (pos < 0 || pos > size) {
			throw new IllegalArgumentException("Invalid position: " + pos);
		}
		this.currentPosition = pos;
	}

	/**
	 * Retourne la position courante d'écriture
	 */
	public int getPosition() {
		return currentPosition;
	}

	/**
	 * Retourne l'espace disponible restant
	 */
	public int getAvailable() {
		return size - currentPosition;
	}

	/**
	 * Ouvre un OutputStream pour écrire dans le chunk
	 */
	public OutputStream appendStream() {
		if (released.get()) {
			throw new IllegalStateException("Chunk is released");
		}
		return new ChunkOutputStream();
	}

	/**
	 * OutputStream pour écrire dans le chunk
	 */
	private class ChunkOutputStream extends OutputStream {
		@Override
		public void write(int b) throws IOException {
			append(new byte[] { (byte) b }, 0, 1);
		}

		@Override
		public void write(byte[] b, int off, int len) throws IOException {
			append(b, off, len);
		}
	}

}
