package net.bluemind.memory.pool.mmap;

import java.io.IOException;
import java.lang.foreign.Arena;
import java.lang.foreign.MemorySegment;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.memory.pool.mmap.MmapSupport.MappedSegment;

public class Segment implements AutoCloseable {
	private static final Logger log = LoggerFactory.getLogger(Segment.class);

	private final long id;
	private final Path filePath;
	private final long size;
	private final MappedSegment memorySegment;
	private final Arena arena;
	private final AtomicLong currentOffset;
	private final AtomicInteger activeChunks;
	private final ReentrantLock offsetLock;

	private volatile boolean closed = false;

	private static final AtomicBoolean MKD_ONCE = new AtomicBoolean(false);

	public Segment(long id, Path baseDir, long size, FdCreator creator, SizeRecorder sizeRecorder) throws IOException {
		this.id = id;
		this.size = size;
		this.currentOffset = new AtomicLong(0);
		this.activeChunks = new AtomicInteger(0);
		this.offsetLock = new ReentrantLock();

		createRootOnce(baseDir);

		this.filePath = baseDir.resolve("segment-" + id + ".dat");

		this.arena = Arena.ofShared();
		this.memorySegment = MmapSupport.mmap(arena, filePath, creator, sizeRecorder, size);
	}

	private void createRootOnce(Path baseDir) throws IOException {
		if (MKD_ONCE.compareAndSet(false, true)) {
			Files.createDirectories(baseDir);
		}
	}

	/**
	 * Alloue un chunk de la taille donnée
	 * 
	 * @param chunkSize taille du chunk à allouer
	 * @return l'offset de début du chunk, ou -1 si pas assez d'espace
	 */
	public long allocate(int chunkSize) {
		if (closed) {
			throw new IllegalStateException("Segment is closed");
		}

		offsetLock.lock();
		try {
			long cur = currentOffset.get();
			long newOffset = cur + chunkSize;
			if (newOffset >= size) {
				log.info("Chunk of {} would overflow {}", chunkSize, this);
				return -1;
			}
			currentOffset.set(newOffset);
			activeChunks.incrementAndGet();
			return cur;
		} finally {
			offsetLock.unlock();
		}
	}

	/**
	 * Libère un chunk
	 */
	public void releaseChunk(long startOffset, int chunkSize) {
		int remaining = activeChunks.decrementAndGet();

		log.debug("Released chunk at offset {} (size {}), {} chunks remaining in segment {}", startOffset, chunkSize,
				remaining, id);
	}

	/**
	 * Retourne le nombre de chunks actifs
	 */
	public int getActiveChunks() {
		return activeChunks.get();
	}

	/**
	 * Retourne l'offset courant (espace utilisé)
	 */
	public long getCurrentOffset() {
		return currentOffset.get();
	}

	/**
	 * Retourne la taille totale du segment
	 */
	public long getSize() {
		return size;
	}

	/**
	 * Retourne l'espace disponible restant
	 */
	public long getAvailableSpace() {
		return size - currentOffset.get();
	}

	/**
	 * Retourne le ratio de remplissage (0.0 à 1.0)
	 */
	public double getFillRatio() {
		return (double) currentOffset.get() / size;
	}

	/**
	 * Retourne le MemorySegment sous-jacent
	 */
	public MemorySegment getMemorySegment() {
		if (closed) {
			throw new IllegalStateException("Segment is closed");
		}
		return memorySegment.seg();
	}

	/**
	 * Retourne l'ID du segment
	 */
	public long getId() {
		return id;
	}

	public boolean canDestroy() {
		return activeChunks.get() == 0 && !closed;
	}

	@Override
	public void close() throws IOException {
		if (closed) {
			return;
		}

		closed = true;

		memorySegment.cleanup();
		arena.close();
	}

	public boolean isClosed() {
		return closed;
	}

	@Override
	public String toString() {
		return "Segment{id=" + id + ", size: " + size + ", used: " + currentOffset.get() + ", chunks: "
				+ activeChunks.get() + ", closed: " + closed + "}";
	}
}
