package net.bluemind.memory.pool.mmap;

import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.util.Deque;
import java.util.Iterator;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Suppliers;

import net.bluemind.memory.pool.api.IMMapPoolConfig;
import net.bluemind.memory.pool.api.MemoryMappedPool;
import net.bluemind.size.helper.MaxMessageSize;

public class MemoryMappedPoolImpl implements MemoryMappedPool {
	private static final Logger logger = LoggerFactory.getLogger(MemoryMappedPoolImpl.class);

	private final IMMapPoolConfig config;
	private final Deque<Segment> segments;
	private final ArrayBlockingQueue<Segment> nextSegments;
	private final AtomicLong nextSegmentId;
	private final ReentrantLock segmentSwitchLock;
	private final ReentrantLock chunkLock;
	private final ReentrantLock cleanupLock;
	private final Condition cleanupCond;
	private final AtomicReference<Segment> activeSegment;
	private volatile boolean closed;
	private final SizeRecorder sizeRecorder;
	private final Supplier<Long> maxChunk;

	public MemoryMappedPoolImpl(IMMapPoolConfig config) {
		this.config = config;
		this.segments = new ConcurrentLinkedDeque<>();
		this.nextSegments = new ArrayBlockingQueue<>(2);
		this.nextSegmentId = new AtomicLong(0);
		this.segmentSwitchLock = new ReentrantLock();
		this.chunkLock = new ReentrantLock();
		this.cleanupLock = new ReentrantLock();
		this.cleanupCond = cleanupLock.newCondition();
		this.closed = false;

		this.sizeRecorder = new SizeRecorder();

		this.activeSegment = new AtomicReference<>();

		backfillSegments();
		backgroundCleanup();
		switchToNextSegment();

		this.maxChunk = Suppliers.memoize(() -> Math.max(MaxMessageSize.get(), config.getMaxChunkSize()));
		logger.info("Created MemoryMappedPool with config: segment-size={}, max-chunk-size={}", config.getSegmentSize(),
				maxChunk.get());

	}

	private void triggerCleanup() {
		cleanupLock.lock();
		try {
			cleanupCond.signal();
		} finally {
			cleanupLock.unlock();
		}
	}

	private void backgroundCleanup() {
		Thread.ofPlatform().name("cleanup")
				.uncaughtExceptionHandler((thread, exp) -> logger.error("CLEANUP failed for {}", thread, exp))
				.start(this::cleanupLoop);
	}

	private void cleanupLoop() {
		while (!closed) {
			cleanupLock.lock();
			try {
				while (!closed) {
					try {
						cleanupCond.await();
						if (!closed) {
							cleanup();
						}
					} catch (InterruptedException e) {
						Thread.currentThread().interrupt();
						return;
					}
				}
				if (closed) {
					break;
				}
			} catch (Throwable t) {// NOSONAR
				logger.error("CLEANUP fail", t);
			} finally {
				cleanupLock.unlock();
			}
		}
		logger.info("CLEANUP leaving loop");
	}

	private void backfillSegments() {
		Thread.ofPlatform().name("prepare-segments")
				.uncaughtExceptionHandler((thread, exp) -> logger.error("BACKFILL failed for {}", thread, exp))
				.start(this::backfillLoop);
	}

	private void backfillLoop() {
		while (!closed) {
			try {
				Segment pending = createSegment();
				boolean accepted = false;
				do {
					accepted = nextSegments.offer(pending, 5, TimeUnit.SECONDS);
					logger.debug("BACKFILL offering accepted: {}, closed: {}", accepted, closed);
				} while (!accepted && !closed);
				if (!accepted) {
					pending.close();
				}
			} catch (IOException e) {
				logger.error("BACKFILL Failed to create segment", e);
				return;
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				logger.error("BACKFILL interrupted", e);
				return;
			} catch (Throwable t) { // NOSONAR
				logger.error("BACKFILL error", t);
			}
		}
		logger.info("BACKFILL leaving loop closed {} ", closed);
	}

	public static MemoryMappedPool create() {
		return new MemoryMappedPoolImpl(MMapPoolConfig.loadDefault());
	}

	public static MemoryMappedPool create(IMMapPoolConfig config) {
		return new MemoryMappedPoolImpl(config);
	}

	private Segment createSegment() throws IOException {
		long id = nextSegmentId.getAndIncrement();
		return new Segment(id, config.getBaseDirectory(), config.getSegmentSize(), chooseCreator(), sizeRecorder);
	}

	private FdCreator chooseCreator() {
		long curLocked = sizeRecorder.lockedInMemory();
		FdCreator creator = null;
		if (curLocked >= config.maxLockedInMemory()) {
			creator = BackingFilesCreators.REAL_FILE;
		} else {
			creator = BackingFilesCreators.MEM_FD;
		}
		return creator;
	}

	/**
	 * Bascule vers le segment suivant si le segment actif est plein
	 */
	private Segment switchToNextSegment() {
		segmentSwitchLock.lock();
		try {
			try {
				Segment tmp = grabSegment();
				activeSegment.set(tmp);
				segments.add(tmp);
				return tmp;
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				return null;
			}
		} finally {
			segmentSwitchLock.unlock();
			triggerCleanup();
		}
	}

	private Segment grabSegment() throws InterruptedException {
		Segment tmp;
		do {
			tmp = nextSegments.poll(1, TimeUnit.SECONDS);
			if (tmp == null) {
				logger.warn("Segment switch is getting slow.");
			}
		} while (tmp == null);
		return tmp;
	}

	private WritableChunk allocateWritableChunk(int maxSize) throws IOException {
		if (closed) {
			throw new IllegalStateException("Pool is closed");
		}

		if (maxSize <= 0 || maxSize > maxChunk.get()) {
			throw new IllegalArgumentException("Invalid chunk size: " + maxSize);
		}

		chunkLock.lock();
		try {
			return allocateWritableLocked(maxSize);
		} catch (IOException ioe) {
			throw ioe;
		} catch (Throwable any) { // NOSONAR
			logger.error("Error in allocation", any);
			throw any;
		} finally {
			chunkLock.unlock();
		}
	}

	private WritableChunk allocateWritableLocked(int maxSize) throws IOException {
		Segment cur = activeSegment.get();
		long offset = cur.allocate(maxSize);

		if (offset == -1) {
			cur = switchToNextSegment();
			if (cur == null) {
				throw new IOException("Could not switch to a new segment");
			}

			offset = cur.allocate(maxSize);
			if (offset == -1) {
				throw new IOException(
						"Failed to allocate chunk of size " + maxSize + " after switch to " + activeSegment);
			}
		}
		return new WritableChunk(activeSegment.get(), config.getSamplePeriod(), offset, maxSize);
	}

	@Override
	public Chunk allocateFrom(InputStream in, int maxSize) throws IOException {
		if (maxSize > maxChunk.get()) {
			throw new IllegalArgumentException("maxSize exceeds max-chunk-size");
		}

		byte[] target = new byte[maxSize];
		int bytesRead = in.read(target);
		ByteBuffer wrapped = ByteBuffer.wrap(target, 0, bytesRead);
		return allocateFrom(wrapped);
	}

	@Override
	public Chunk allocateFrom(byte[] bytes) throws IOException {
		WritableChunk chunk = allocateWritableChunk(bytes.length);
		chunk.append(bytes);
		return chunk;
	}

	@Override
	public Chunk allocateFrom(ByteBuffer bytes) throws IOException {
		int size = bytes.remaining();
		WritableChunk chunk = allocateWritableChunk(size);
		chunk.append(bytes);
		return chunk;
	}

	@Override
	public WritableChunk allocateEmpty(int maxSize) throws IOException {
		return allocateWritableChunk(maxSize);
	}

	@Override
	public PoolStats getStats() {
		int totalSegments = segments.size();
		int activeSegments = 0;
		long totalSize = 0;
		long usedSize = 0;
		int totalChunks = 0;

		for (Segment segment : segments) {
			if (!segment.isClosed()) {
				activeSegments++;
				totalSize += segment.getSize();
				usedSize += segment.getCurrentOffset();
				totalChunks += segment.getActiveChunks();
			}
		}

		return new PoolStats(totalSegments, activeSegments, totalSize, usedSize, totalChunks,
				activeSegment.get().getId());
	}

	@Override
	public void cleanup() {
		Iterator<Segment> iterator = segments.iterator();

		while (iterator.hasNext()) {
			Segment segment = iterator.next();
			if (iterator.hasNext() && segment.canDestroy()) {
				try {
					segment.close();
					iterator.remove();
					logger.info("Cleaned up segment {}", segment.getId());
				} catch (IOException e) {
					logCloseFailed(segment, e);
				}
			}
		}
	}

	@Override
	public void close() throws IOException {
		if (closed) {
			return;
		}

		closed = true;
		logger.info("Closing MemoryMappedPool {}", this);
		triggerCleanup();

		IOException firstException = null;
		for (Segment segment : segments) {
			try {
				segment.close();
			} catch (IOException e) {
				if (firstException == null) {
					firstException = e;
				}
				logCloseFailed(segment, e);
			}
		}
		segments.clear();
		for (Segment segment : nextSegments) {
			try {
				segment.close();
			} catch (IOException e) {
				if (firstException == null) {
					firstException = e;
				}
				logCloseFailed(segment, e);
			}
		}
		nextSegments.clear();

		if (firstException != null) {
			throw firstException;
		}

		logger.info("MemoryMappedPool {} closed", this);
	}

	private void logCloseFailed(Segment segment, IOException e) {
		logger.error("Failed to close segment {}", segment.getId(), e);
	}

	@Override
	public String toString() {
		return String.format("MemoryMappedPool[segments=%d, active=%d, stats=%s]", segments.size(),
				activeSegment.get().getId(), getStats());
	}
}