package net.bluemind.memory.pool.mmap;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;

import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;

import net.bluemind.memory.pool.api.IMMapPoolConfig;

public class MMapPoolConfig implements IMMapPoolConfig {
	private final long segmentSize;
	private final long maxChunkSize;
	private final Path baseDirectory;
	private final boolean enableStats;
	private final long maxLockedInMemory;
	private final int samplePeriod;

	public MMapPoolConfig(Config config) {
		Config poolConfig = config.getConfig("mmap-pool");

		this.segmentSize = poolConfig.getBytes("segment-size");
		this.maxChunkSize = poolConfig.getBytes("max-chunk-size");
		this.baseDirectory = Paths.get(poolConfig.getString("base-directory"));
		this.enableStats = poolConfig.getBoolean("enable-stats");
		this.maxLockedInMemory = poolConfig.getBytes("max-locked-memory");
		this.samplePeriod = poolConfig.getInt("sample-every-n-alloc");

		validate();
	}

	public static MMapPoolConfig loadDefault() {
		return load("resources/application.conf");
	}

	public static MMapPoolConfig load(String resourceName) {
		Config config = ConfigFactory.load(MMapPoolConfig.class.getClassLoader(), resourceName);
		String path = "/etc/bm/" + System.getProperty("net.bluemind.property.product", "all") + "-memory-pool.conf";
		File local = new File(path);
		if (local.exists()) {
			Config fromFile = ConfigFactory.parseFile(local);
			config = fromFile.withFallback(config);
		}
		return new MMapPoolConfig(config);
	}

	private void validate() {
		if (segmentSize <= 0) {
			throw new IllegalArgumentException("segment-size must be positive");
		}
		if (maxChunkSize <= 0 || maxChunkSize > segmentSize) {
			throw new IllegalArgumentException("max-chunk-size must be positive and <= segment-size");
		}
	}

	@Override
	public long maxLockedInMemory() {
		return maxLockedInMemory;
	}

	public int getSamplePeriod() {
		return samplePeriod;
	}

	@Override
	public long getSegmentSize() {
		return segmentSize;
	}

	@Override
	public long getMaxChunkSize() {
		return maxChunkSize;
	}

	@Override
	public Path getBaseDirectory() {
		return baseDirectory;
	}

	@Override
	public boolean isEnableStats() {
		return enableStats;
	}
}