package net.bluemind.memory.pool.mmap;

import java.io.InputStream;
import java.lang.foreign.MemorySegment;
import java.lang.ref.Cleaner;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.Unpooled;
import net.bluemind.memory.pool.api.IChunk;

public class Chunk implements IChunk {
	protected final Segment parent;
	protected final long startOffset;
	protected final int size;
	protected final AtomicBoolean released;

	private static final AtomicLong allocated = new AtomicLong();
	private static final Cleaner BY_GC = Cleaner.create(Thread.ofVirtual().factory());
	private static final Logger logger = LoggerFactory.getLogger(Chunk.class);

	private static record UnreleasedChunk(AtomicBoolean rel, Segment parent, long start, int size, Throwable alloc)
			implements Runnable {

		@Override
		public void run() {
			if (rel.compareAndSet(false, true)) {
				parent.releaseChunk(start, size);
				if (alloc != null) {
					logger.warn("Chunk was reclaimed by GC instead of being released properly, please report", alloc);
				}
			}
		}

	}

	public Chunk(Segment parent, int samplePeriod, long startOffset, int size) {
		this.parent = parent;
		this.startOffset = startOffset;
		this.size = size;
		this.released = new AtomicBoolean(false);

		Throwable alloc = null;
		if (allocated.incrementAndGet() % samplePeriod == 0) {
			alloc = new Throwable("leaked-chunk").fillInStackTrace();
		}
		UnreleasedChunk unrel = new UnreleasedChunk(released, parent, startOffset, size, alloc);
		BY_GC.register(this, unrel);
	}

	public void release() {
		if (released.compareAndSet(false, true)) {
			parent.releaseChunk(startOffset, size);
		}
	}

	public boolean isReleased() {
		return released.get();
	}

	public ByteBuf openBuffer() {
		if (released.get()) {
			throw new IllegalStateException("Chunk is released");
		}

		MemorySegment segment = parent.getMemorySegment();
		MemorySegment slice = segment.asSlice(startOffset, getSize());

		java.nio.ByteBuffer nioBuffer = slice.asByteBuffer();

		return Unpooled.wrappedBuffer(nioBuffer).asReadOnly();
	}

	public InputStream openStream() {
		if (released.get()) {
			throw new IllegalStateException("Chunk is released");
		}
		ByteBuf in = openBuffer();
		return new ByteBufInputStream(in, true);
	}

	public int getSize() {
		return size;
	}

	public long getStartOffset() {
		return startOffset;
	}

	public Segment getParent() {
		return parent;
	}

	@Override
	public String toString() {
		return "Chunk{seg: " + parent.getId() + ", offset: " + startOffset + ", size: " + getSize() + ", released: "
				+ released.get() + "}";
	}
}