package net.bluemind.backend.mail.replica.service.internal;

import java.io.InputStream;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.apache.james.mime4j.dom.Header;
import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.address.AddressList;
import org.apache.james.mime4j.dom.address.MailboxList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufInputStream;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import net.bluemind.addressbook.api.IAddressBook;
import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.addressbook.api.IAddressBooks;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.api.VCard.Communications.Email;
import net.bluemind.authentication.api.AuthUser;
import net.bluemind.authentication.api.IAuthentication;
import net.bluemind.backend.mail.api.IItemsTransfer;
import net.bluemind.backend.mail.api.IMailboxFolders;
import net.bluemind.backend.mail.api.IMailboxFoldersByContainer;
import net.bluemind.backend.mail.api.IMailboxItems;
import net.bluemind.backend.mail.api.IOutbox;
import net.bluemind.backend.mail.api.MailboxFolder;
import net.bluemind.backend.mail.api.MailboxItem;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.flags.FlagUpdate;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.InCoreMailboxRecords;
import net.bluemind.backend.mail.replica.api.MailApiHeaders;
import net.bluemind.backend.mail.replica.service.deferredaction.ScheduleMailDeferredAction;
import net.bluemind.backend.mail.replica.service.internal.tools.EnvelopFrom;
import net.bluemind.config.Token;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.SortDescriptor;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.container.service.internal.RBACManager;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.sanitizer.Sanitizer;
import net.bluemind.core.sendmail.ISendmail;
import net.bluemind.core.task.api.ITask;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.core.task.service.ITasksManager;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.deferredaction.api.DeferredAction;
import net.bluemind.deferredaction.api.IDeferredActionContainerUids;
import net.bluemind.deferredaction.api.IInternalDeferredAction;
import net.bluemind.deferredaction.registry.DeferredActionExecution;
import net.bluemind.delivery.smtp.ndr.SendmailCredentials;
import net.bluemind.delivery.smtp.ndr.SendmailHelper;
import net.bluemind.delivery.smtp.ndr.SendmailResponse;
import net.bluemind.delivery.smtp.ndr.SendmailResponseManagement;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.lifecycle.helper.SoftReset;
import net.bluemind.mailbox.api.IMailboxAclUids;
import net.bluemind.mailbox.api.Mailbox;
import net.bluemind.mime4j.common.Mime4JHelper;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public class OutboxService implements IOutbox {

	private static final Logger logger = LoggerFactory.getLogger(OutboxService.class);
	private static final int MAX_SCHEDULE_RETRY = 12;

	private final BmContext context;
	private final String domainUid;
	private final ItemValue<Mailbox> mailboxItem;
	private final IServiceProvider serviceProvider;
	private final RBACManager rbac;
	private final Sanitizer sortDescSanitizer;

	private ISendmail mailer;

	private static final Map<String, TaskRef> ONCE_PER_OWNER = buildPerOwnerProtection();

	private static Map<String, TaskRef> buildPerOwnerProtection() {
		ConcurrentHashMap<String, TaskRef> active = new ConcurrentHashMap<>();
		SoftReset.register(active::clear);

		return active;
	}

	public OutboxService(BmContext context, String domainUid, ItemValue<Mailbox> mailboxItem, ISendmail mailer) {
		this.context = context;
		this.domainUid = domainUid;
		this.mailboxItem = mailboxItem;
		this.serviceProvider = context.provider();
		this.rbac = RBACManager.forContext(context).forContainer(IMailboxAclUids.uidForMailbox(mailboxItem.uid));
		this.sortDescSanitizer = new Sanitizer(context);

		this.mailer = mailer;
	}

	@Override
	public TaskRef flush() {
		rbac.check(Verb.Write.name());

		return ONCE_PER_OWNER.compute(mailboxItem.uid, (owner, tsk) -> {
			if (tsk == null || isFinished(tsk)) {
				return flushOncePerOwner();
			} else {
				return tsk;
			}
		});
	}

	private boolean isFinished(TaskRef tsk) {
		try {
			ITask taskApi = context.su().provider().instance(ITask.class, tsk.id);
			return taskApi.status().state.ended;
		} catch (Exception e) {
			// should be ServerFault NOT_FOUND
			return true;
		}
	}

	private TaskRef flushOncePerOwner() {
		return serviceProvider.instance(ITasksManager.class).run(m -> BlockingServerTask.run(m, monitor -> {

			ItemValue<MailboxFolder> outboxFolder = getOutboxFolder();

			List<ItemValue<MailboxItem>> mails = retrieveOutboxItems(outboxFolder.uid);

			List<ItemValue<MailboxItem>> notScheduled = mails.stream().filter(item -> mustBeSent(item.value)).toList();

			FlushContext ctx = buildFlush(monitor, outboxFolder);
			if (ctx != null) {
				Set<Long> sentMailIds = flushAll(monitor, notScheduled, ctx);
				scheduleMail(mails.stream().filter(mail -> !sentMailIds.contains(mail.internalId)).toList());
			}
		}));
	}

	private ItemValue<MailboxFolder> getOutboxFolder() {
		IMailboxFolders mailboxFoldersService = serviceProvider.instance(IMailboxFoldersByContainer.class,
				IMailReplicaUids.subtreeUid(domainUid, mailboxItem));
		return mailboxFoldersService.byName("Outbox");
	}

	private FlushContext buildFlush(IServerTaskMonitor monitor, ItemValue<MailboxFolder> outboxFolder) {
		try {
			InCoreMailboxRecords inCore = (InCoreMailboxRecords) serviceProvider.instance(IDbMailboxRecords.class,
					outboxFolder.uid);
			IMailboxFolders mailboxFoldersService = serviceProvider.instance(IMailboxFoldersByContainer.class,
					IMailReplicaUids.subtreeUid(domainUid, mailboxItem));
			ItemValue<MailboxFolder> sentFolder = mailboxFoldersService.byName("Sent");

			AuthUser user = serviceProvider.instance(IAuthentication.class).getCurrentUser();
			return new FlushContext(monitor, outboxFolder, sentFolder, inCore, user);

		} catch (Exception e) {
			logger.error("Error while building flush context for outbox {} ", outboxFolder.uid, e);
			return null;
		}
	}

	private void scheduleMail(List<ItemValue<MailboxItem>> mails) {
		Optional<Long> nextMailToSchedule = mails.stream().map(item -> scheduledHeaderValue(item.value))
				.filter(this::isValid).map(Optional::get).sorted().findFirst();
		Optional<ItemValue<DeferredAction>> mailScheduler = getMailScheduler();

		if (nextMailToSchedule.isEmpty() && mailScheduler.isPresent()) {
			deleteMailScheduler(mailScheduler.get());
		} else if (nextMailToSchedule.isPresent() && mailScheduler.isEmpty()) {
			createMailScheduler(nextMailToSchedule.get());
		} else if (nextMailToSchedule.isPresent() && !isScheduled(nextMailToSchedule.get(), mailScheduler)) {
			createMailScheduler(nextMailToSchedule.get());
			deleteMailScheduler(mailScheduler.get());
		}
	}

	private boolean isValid(Optional<Long> timestamp) {

		long stopRetry = Instant.now().minus(MAX_SCHEDULE_RETRY * DeferredActionExecution.PERIOD, ChronoUnit.MILLIS)
				.toEpochMilli();
		return timestamp.map(ts -> ts > stopRetry).orElse(false);
	}

	private boolean isScheduled(Long mail, Optional<ItemValue<DeferredAction>> nextScheduledAction) {
		long mailScheduledTime = ScheduleMailDeferredAction.getExecutionDate(mail).getTime();
		return nextScheduledAction.isPresent()
				&& mailScheduledTime == (nextScheduledAction.get().value.executionDate.getTime());
	}

	private void createMailScheduler(Long executionTimestamp) {
		IInternalDeferredAction deferredActionService = getDeferredActionService();
		DeferredAction action = new DeferredAction();
		action.reference = ScheduleMailDeferredAction.reference(mailboxItem);
		action.executionDate = ScheduleMailDeferredAction.getExecutionDate(executionTimestamp);
		action.actionId = ScheduleMailDeferredAction.ACTION_ID;
		deferredActionService.create(action);
	}

	private void deleteMailScheduler(ItemValue<DeferredAction> itemValue) {
		IInternalDeferredAction deferredActionService = getDeferredActionService();
		deferredActionService.delete(itemValue.uid);
	}

	private Optional<ItemValue<DeferredAction>> getMailScheduler() {
		IInternalDeferredAction deferredActionService = getDeferredActionService();

		List<ItemValue<DeferredAction>> deferredActions = deferredActionService
				.getByReference(ScheduleMailDeferredAction.reference(mailboxItem));
		if (deferredActions.size() > 1) {
			logger.warn("There is more than one deferred action for mailbox {}", mailboxItem.displayName);
		}
		return deferredActions.stream().findFirst();

	}

	private IInternalDeferredAction getDeferredActionService() {
		ServerSideServiceProvider provider = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		String deferredActionUid = IDeferredActionContainerUids.uidForDomain(domainUid);
		IInternalDeferredAction deferredActionService = provider.instance(IInternalDeferredAction.class,
				deferredActionUid);
		return deferredActionService;
	}

	private Optional<Long> scheduledHeaderValue(MailboxItem item) {
		Optional<net.bluemind.backend.mail.api.MessageBody.Header> header = item.body.headers.stream()
				.filter(h -> h.name.equals(MailApiHeaders.X_BM_DRAFT_SCHEDULE)).findFirst();
		return header.map(h -> {
			try {
				long timestamp = Long.parseLong(header.get().firstValue());
				return timestamp;
			} catch (NumberFormatException e) {
				return null;
			}
		});
	}

	private boolean mustBeSent(MailboxItem item) {
		if (item.body.headers.stream()
				.noneMatch(header -> header.name.equals(MailApiHeaders.X_BM_DRAFT_REFRESH_DATE))) {
			return false;
		}
		return scheduledHeaderValue(item).orElse(0L) <= Instant.now().toEpochMilli();
	}

	private Set<Long> flushAll(IServerTaskMonitor monitor, List<ItemValue<MailboxItem>> mails, FlushContext ctx) {
		int mailCount = mails.size();
		monitor.begin(mailCount, "FLUSHING OUTBOX - have " + mailCount + "mails to send.");
		logger.info("[{}] Flushing {} outbox item(s).", context.getSecurityContext().getSubject(), mailCount);

		List<FlushInfo> promises = mails.stream().map(item -> flushOne(ctx, item)).toList();

		List<FlushResult> flushResults = promises.stream() //
				.filter(ret -> ret.flushResult.isPresent()) //
				.map(ret -> ret.flushResult.get()) //
				.toList();
		Set<RecipientInfo> collectedRecipients = promises.stream() //
				.flatMap(ret -> ret.collectedRecipients.stream()).collect(Collectors.toSet());
		int requestedDSNs = promises.stream().map(fi -> fi.requestedDSN ? 1 : 0).reduce(0, (a, b) -> a + b);

		addRecipientsToCollectedContacts(ctx.user.uid, collectedRecipients);
		logger.debug("[{}] flushed {}", context.getSecurityContext().getSubject(), mailCount);
		monitor.end(true, "FLUSHING OUTBOX finished successfully", String
				.format("{\"result\": %s, \"requestedDSNs\": %d}", JsonUtils.asString(flushResults), requestedDSNs));
		return flushResults.stream().map(fi -> fi.sourceInternalId).collect(Collectors.toSet());
	}

	private static final byte[] END_OF_HEADERS = "\r\n\r\n".getBytes();

	private FlushInfo flushOne(FlushContext ctx, ItemValue<MailboxItem> item) {
		ByteBuf buf = ctx.inCore.fetchByGuid(item.value.body.guid);
		int endOfHeaders = ByteBufUtil.indexOf(Unpooled.wrappedBuffer(END_OF_HEADERS), buf);
		if (endOfHeaders <= 0) {
			throw new ServerFault("ItemId " + item.internalId + " does not have a valid header");
		}
		ByteBuf fullEml = buf.duplicate();
		ByteBuf withHeader = fullEml.readSlice(endOfHeaders + END_OF_HEADERS.length);
		ByteBuf headerLess = fullEml;
		InputStream headersInputStream = new ByteBufInputStream(withHeader);
		FlushInfo ret = new FlushInfo();
		try (Message msg = Mime4JHelper.parse(headersInputStream, false)) {
			if (msg.getFrom() == null) {
				org.apache.james.mime4j.dom.address.Mailbox fromCtx = SendmailHelper.formatAddress(ctx.user.displayName,
						ctx.user.value.defaultEmail().address);
				msg.setFrom(fromCtx);
			}
			String fromMail = msg.getFrom().iterator().next().getAddress();
			MailboxList rcptTo = allRecipients(msg);
			ByteBuf clearedHeader = filterUnwantedHeaders(msg);
			ByteBuf freshEml = Unpooled.wrappedBuffer(clearedHeader, headerLess);
			InputStream forSend = new ByteBufInputStream(freshEml);
			SendmailResponse sendmailResponse = send(ctx.user.value, forSend, fromMail, rcptTo, msg,
					requestDSN(item.value));
			ret.requestedDSN = sendmailResponse.getRequestedDSNs() > 0;
			boolean moveToSent = !isMDN(item.value);
			ret.flushResult = moveToSent ? moveToSent(item, ctx.sentFolder, ctx.outboxFolder)
					: Optional.ofNullable(remove(item, ctx.outboxFolder));
			ret.collectedRecipients = rcptTo.stream()
					.map(rcpt -> new RecipientInfo(rcpt.getAddress(), rcpt.getName(), rcpt.getLocalPart()))
					.collect(Collectors.toSet());
			if (moveToSent) {
				ctx.monitor.progress(1,
						String.format("FLUSHING OUTBOX - mail %s sent and moved in Sent folder.", msg.getMessageId()));
			} else {
				ctx.monitor.progress(1, String.format("FLUSHING OUTBOX - mail %s sent. Requested DSN: %b",
						msg.getMessageId(), ret.requestedDSN));
			}
		} catch (ServerFault sf) {
			addFlagToMailInFailure(ctx, item.internalId);
			throw sf;
		} catch (Exception e) {
			addFlagToMailInFailure(ctx, item.internalId);
			throw new ServerFault("ItemId " + item.internalId, e);
		}
		return ret;
	}

	private void addFlagToMailInFailure(FlushContext ctx, long internalId) {
		serviceProvider.instance(IMailboxItems.class, ctx.outboxFolder.uid)
				.addFlag(FlagUpdate.of(internalId, new MailboxItemFlag("BmSendFailure")));
	}

	/**
	 * @param message
	 * @return the header buffer including the <code>2*CRLF</code> separator
	 */
	private ByteBuf filterUnwantedHeaders(Message message) {
		Header toFilter = message.getHeader();
		// getFields() returns an unmodifiable list, removeIf is not usable
		List.copyOf(toFilter.getFields()).forEach(field -> {
			if (field.getName().toLowerCase().startsWith("x-bm-draft")) {
				toFilter.removeFields(field.getName());
			}
		});

		ByteBufOutputStream out = new ByteBufOutputStream(Unpooled.buffer());
		Mime4JHelper.serialize(message, out);
		ByteBuf serializedHeaders = out.buffer();
		// we slice it again as mime4j may produce a part boundary for multipart
		// messages
		int endOfHeaders = ByteBufUtil.indexOf(Unpooled.wrappedBuffer(END_OF_HEADERS), serializedHeaders);
		return endOfHeaders > 0 ? serializedHeaders.slice(0, endOfHeaders + END_OF_HEADERS.length) : serializedHeaders;

	}

	/**
	 * @return <code>true</code> if <code>mailboxItem</code> is a Message
	 *         Disposition Notification (rfc8098)
	 */
	private boolean isMDN(MailboxItem mailboxItem) {
		return "multipart/report".equalsIgnoreCase(mailboxItem.body.structure.mime)
				&& mailboxItem.body.structure.children.stream()
						.anyMatch(child -> child.mime.contains("/disposition-notification"));
	}

	/**
	 * @return <code>true</code> if a Delivery Status Notification is requested for
	 *         <code>mailboxItem</code> (rfc1891)
	 */
	private boolean requestDSN(MailboxItem mailboxItem) {
		return mailboxItem.flags.stream().anyMatch(itemFlag -> "BmDSN".equalsIgnoreCase(itemFlag.flag));
	}

	private void addRecipientsToCollectedContacts(String uid, Set<RecipientInfo> collectedRecipients) {
		IAddressBooks allContactsService = serviceProvider.instance(IAddressBooks.class);
		IAddressBook collectedContactsService = serviceProvider.instance(IAddressBook.class,
				IAddressBookUids.collectedContactsUserAddressbook(uid));
		collectedRecipients.forEach(recipient -> {
			try {
				if (allContactsService.findUidsByEmail(recipient.email).isEmpty()) {
					addRecipientToCollectedContacts(collectedContactsService, recipient);
				}
			} catch (ServerFault sf) {
				logger.error("IAddressBooks.findUidsByEmail({}) failed", recipient.email, sf);
			}
		});
	}

	private void addRecipientToCollectedContacts(IAddressBook service, RecipientInfo recipient) {
		VCard card = recipientToVCard(recipient);
		service.create(UUID.randomUUID().toString(), card);
	}

	private VCard recipientToVCard(RecipientInfo recipient) {
		VCard card = new VCard();

		card.identification.name = VCard.Identification.Name.create(recipient.familyNames, recipient.givenNames, null,
				null, null, null);

		List<Email> emails = Arrays.asList(VCard.Communications.Email.create(recipient.email));
		card.communications.emails = emails;
		return card;
	}

	private SendmailResponse send(User user, InputStream forSend, String fromMail, MailboxList rcptTo,
			Message relatedMsg, boolean requestDSN) {
		SendmailCredentials creds = SendmailCredentials.as(String.format("%s@%s", user.login, domainUid),
				Token.admin0());

		ItemValue<Domain> domain = serviceProvider.instance(IInCoreDomains.class).getUnfiltered(domainUid);
		String from = new EnvelopFrom(domain).getFor(creds, user, fromMail);
		SendmailResponse sendmailResponse = mailer.send(creds, from, domainUid, rcptTo, forSend, requestDSN);
		if (sendmailResponse.isIOError()) {
			throw new ServerFault("Error while connecting to SMTP server: " + sendmailResponse.getMessage());
		}
		if (sendmailResponse.isFailedResponse()) {
			sendNdrMessage(user, relatedMsg, creds.isAdminO(), domain.value, sendmailResponse);
		}
		return sendmailResponse;
	}

	private void sendNdrMessage(User user, Message relatedMsg, boolean isAdmin0, Domain domain,
			SendmailResponse sendmailResponse) {
		if (notAdminAndNotCurrentUser(isAdmin0, sendmailResponse.getOriginalFrom(), user, domain)) {
			sendmailResponse.setOriginalSender(user.defaultEmailAddress());
		}
		Locale locale = getUserRcptLocale(sendmailResponse.getOriginalFrom());
		new SendmailResponseManagement(sendmailResponse, domain.defaultAlias, relatedMsg, locale).createNdrMessage()
				.ifPresent(ndrMsg -> {
					String fromAddress = ndrMsg.message().getFrom().getFirst().getAddress();
					mailer.send(ndrMsg.creds(), fromAddress, domain.defaultAlias, ndrMsg.message().getTo().flatten(),
							ndrMsg.message());
				});
	}

	private boolean notAdminAndNotCurrentUser(boolean isAdmin0, String sender, User user, Domain domain) {
		return !isAdmin0 && user.emails.stream().noneMatch(e -> e.match(sender, domain.aliases));
	}

	private Locale getUserRcptLocale(String email) {
		IUser userService = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IUser.class,
				domainUid);
		ItemValue<User> userEntry = userService.byEmail(email);
		return Locale.of(userService.getLocale(userEntry.uid));
	}

	/**
	 * Move to default Sent folder or the one given in the X-BM-DRAFT-SENT-FOLDER
	 * header. Fall back to default Sent folder if an error occurs. <br/>
	 * Remove from outbox if cannot be moved to default Sent folder
	 */
	private Optional<FlushResult> moveToSent(ItemValue<MailboxItem> item, ItemValue<MailboxFolder> sentFolder,
			ItemValue<MailboxFolder> outboxFolder) {
		Optional<String> xBmDraftSentFolder = extractXBmDraftSentFolder(item);
		FlushResult moveFlushResult = null;

		boolean moveToSeparateSentFolder = xBmDraftSentFolder.map(folder -> !folder.equals(sentFolder.uid))
				.orElse(false);
		if (moveToSeparateSentFolder) {
			try {
				moveFlushResult = moveTo(item, outboxFolder.uid, xBmDraftSentFolder.get());
			} catch (ServerFault e) {
				logger.warn(
						"Could not move sent message (ItemId[{}]) to separate Sent folder {}, fall back to default Sent folder.",
						item.uid, xBmDraftSentFolder.get());
				moveFlushResult = moveTo(item, outboxFolder.uid, sentFolder.uid);
			}
		} else {
			try {
				moveFlushResult = moveTo(item, outboxFolder.uid, sentFolder.uid);
			} catch (ServerFault e) {
				logger.warn("Could not move sent message (ItemId[{}]) to Sent folder.", item.uid);
			}
		}

		if (moveFlushResult == null) {
			logger.error("Could not move sent message (ItemId[{}]) to folder, force removal from outbox.", item.uid);
			remove(item, outboxFolder);
		}

		return Optional.ofNullable(moveFlushResult);
	}

	private FlushResult moveTo(ItemValue<MailboxItem> item, String sourceUid, String targetUid) {
		IItemsTransfer itemsTransferService = serviceProvider.instance(IItemsTransfer.class, sourceUid, targetUid);
		List<ItemIdentifier> targetItems = itemsTransferService.move(Arrays.asList(item.internalId));
		if (targetItems == null || targetItems.isEmpty()) {
			return null;
		}
		return new FlushResult(item.internalId, sourceUid, targetItems.get(0).id, targetUid);
	}

	private FlushResult remove(ItemValue<MailboxItem> item, ItemValue<MailboxFolder> outboxFolder) {
		IMailboxItems mailboxItemsService = serviceProvider.instance(IMailboxItems.class, outboxFolder.uid);
		mailboxItemsService.deleteById(item.internalId);
		return new FlushResult(item.internalId, outboxFolder.uid, item.internalId, outboxFolder.uid);
	}

	private static final String X_BM_DRAFT_SENT_FOLDER_HEADER = "x-bm-draft-sent-folder";
	private static final String OLD_X_BM_DRAFT_SENT_FOLDER_HEADER = "x-bm-sent-folder";

	private Optional<String> extractXBmDraftSentFolder(ItemValue<MailboxItem> item) {
		return item.value.body.headers.stream()
				.filter(header -> header.name.equalsIgnoreCase(X_BM_DRAFT_SENT_FOLDER_HEADER)
						|| header.name.equalsIgnoreCase(OLD_X_BM_DRAFT_SENT_FOLDER_HEADER))
				.findFirst().map(MessageBody.Header::firstValue);
	}

	private MailboxList allRecipients(Message m) {
		LinkedList<org.apache.james.mime4j.dom.address.Mailbox> rcpt = new LinkedList<>();
		AddressList tos = m.getTo();
		if (tos != null) {
			rcpt.addAll(tos.flatten());
		}
		AddressList ccs = m.getCc();
		if (ccs != null) {
			rcpt.addAll(ccs.flatten());
		}
		AddressList bccs = m.getBcc();
		if (bccs != null) {
			rcpt.addAll(bccs.flatten());
		}
		return new MailboxList(rcpt, true);
	}

	private List<ItemValue<MailboxItem>> retrieveOutboxItems(String outboxUid) {
		IMailboxItems mailboxItemsService = serviceProvider.instance(IMailboxItems.class, outboxUid);
		long enumerate = System.currentTimeMillis();

		SortDescriptor sortDescriptor = new SortDescriptor();
		sortDescSanitizer.create(sortDescriptor);

		List<Long> mailsIds = mailboxItemsService.sortedIds(sortDescriptor);
		List<ItemValue<MailboxItem>> listMailboxItems = new ArrayList<>(mailsIds.size());
		for (List<Long> slice : Lists.partition(mailsIds, 250)) {
			listMailboxItems.addAll(mailboxItemsService.multipleGetById(slice));
		}

		enumerate = System.currentTimeMillis() - enumerate;
		logger.info("[{}] Flushing outbox retrieve {} item(s), took {}ms.", context.getSecurityContext().getSubject(),
				listMailboxItems.size(), enumerate);
		return listMailboxItems;
	}

	static class RecipientInfo {
		final String email;
		final String givenNames;
		final String familyNames;

		public RecipientInfo(String email, String fullName, String localPart) {
			this.email = email;

			String[] names;
			if (fullName == null) {
				names = Arrays.asList(localPart.split("\\.")).stream().map(this::captitalize).toArray(String[]::new);
			} else {
				names = fullName.split(" ");
			}
			this.givenNames = names[0];
			this.familyNames = String.join(" ", Arrays.copyOfRange(names, 1, names.length));
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((email == null) ? 0 : email.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj)
				return true;
			if (obj == null)
				return false;
			if (getClass() != obj.getClass())
				return false;
			RecipientInfo other = (RecipientInfo) obj;
			if (email == null) {
				if (other.email != null)
					return false;
			} else if (!email.equals(other.email))
				return false;
			return true;
		}

		private String captitalize(String str) {
			return Character.toUpperCase(str.charAt(0)) + str.substring(1);
		}

	}

	static class FlushContext {
		final IServerTaskMonitor monitor;
		final ItemValue<MailboxFolder> outboxFolder;
		final ItemValue<MailboxFolder> sentFolder;
		final AuthUser user;
		final InCoreMailboxRecords inCore;

		public FlushContext(IServerTaskMonitor monitor, ItemValue<MailboxFolder> outboxFolder,
				ItemValue<MailboxFolder> sentFolder, InCoreMailboxRecords inCore, AuthUser user) {
			this.monitor = monitor;
			this.outboxFolder = outboxFolder;
			this.sentFolder = sentFolder;
			this.inCore = inCore;
			this.user = user;
		}
	}

	static class FlushInfo {
		public boolean requestedDSN;
		Optional<FlushResult> flushResult;
		Set<RecipientInfo> collectedRecipients;
	}

	private record FlushResult(long sourceInternalId, String sourceFolderUid, long destinationInternalId,
			String destinationFolderUid) {
	}

}
