/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2024
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
*/
package net.bluemind.resource.service.event;

import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.UUID;

import net.bluemind.calendar.api.VEvent;
import net.bluemind.calendar.api.VEventOccurrence;
import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.calendar.occurrence.OccurrenceHelper;
import net.bluemind.core.api.date.BmDateTime;
import net.bluemind.core.api.date.BmDateTime.Precision;
import net.bluemind.core.api.date.BmDateTimeWrapper;
import net.bluemind.core.container.model.ItemValue;

public class Recurrency {
	private static final long NB_DAYS_ALLOWED = 365;
	private final ZonedDateTime now;
	private final VEvent vEvent;
	private final Precision precision;

	private BmDateTime startDate;
	private BmDateTime endDate;
	private boolean valid;
	private int days = 0;

	public Recurrency(VEvent event) {
		this.vEvent = event;
		now = ZonedDateTime.now();
		this.valid = false;
		precision = event.allDay() ? Precision.Date : Precision.DateTime;
		calculateDates();
	}

	public boolean isValid() {
		return valid;
	}

	public static long getNbDaysRecurrency() {
		return NB_DAYS_ALLOWED;
	}

	public BmDateTime getStartDate() {
		return this.startDate;
	}

	public BmDateTime getEndDate() {
		return endDate;
	}

	private void calculateDates() {
		if (vEvent.rrule.until != null) {
			endDate = vEvent.rrule.until;
		} else {
			if (vEvent.rrule.count != null) {
				calculateEndDateFromCount();
			} else {
				valid = false;
				return;
			}
		}

		ZonedDateTime zdtStart = fromBmDateTime(vEvent.dtstart);
		ZonedDateTime zdtEnd = fromBmDateTime(endDate);

		// If recurrency from the past, allow for 1 year from now
		if (zdtStart.isBefore(now)) {
			this.startDate = toBmDatetime(now);
			calculateDays(now, zdtEnd);
		} else {
			this.startDate = vEvent.dtstart;
			calculateDays(zdtStart, zdtEnd);
		}

		this.valid = (days <= NB_DAYS_ALLOWED);
	}

	private void calculateEndDateFromCount() {
		ZonedDateTime zdtStart = fromBmDateTime(vEvent.dtstart);
		BmDateTime periodStart = toBmDatetime(zdtStart);
		if (zdtStart.isBefore(now)) {
			periodStart = toBmDatetime(now);
		}

		ZonedDateTime nowPlusAYear = now.plusDays(NB_DAYS_ALLOWED);
		BmDateTime periodEnd = toBmDatetime(nowPlusAYear);
		this.endDate = periodEnd;

		VEventSeries serie = new VEventSeries();
		serie.main = this.vEvent;
		List<VEvent> occurrencesInAYear = OccurrenceHelper.list(ItemValue.create(UUID.randomUUID().toString(), serie),
				periodStart, periodEnd);

		if (occurrencesInAYear != null && !occurrencesInAYear.isEmpty()) {
			VEvent lastOccurenceInAYear = occurrencesInAYear.getLast();

			// occurrences after a year
			Optional<VEventOccurrence> nextOccurence = OccurrenceHelper.getNextOccurrence(lastOccurenceInAYear.dtend,
					this.vEvent);

			if (nextOccurence.isEmpty()) {
				endDate = lastOccurenceInAYear.dtend;
			} else {
				while (nextOccurence.isPresent()) {
					endDate = nextOccurence.get().dtend;
					nextOccurence = OccurrenceHelper.getNextOccurrence(nextOccurence.get().dtstart, vEvent);
				}
			}
		}
	}

	private ZonedDateTime fromBmDateTime(BmDateTime bdt) {
		return new BmDateTimeWrapper(bdt).toDateTime();
	}

	private BmDateTime toBmDatetime(ZonedDateTime zdt) {
		return BmDateTimeWrapper.create(zdt, precision);
	}

	private void calculateDays(ZonedDateTime start, ZonedDateTime end) {
		this.days = (int) ChronoUnit.DAYS.between(start, end);
	}

}
