/* BEGIN LICENSE
* Copyright © Blue Mind SAS, 2012-2024
*
* This file is part of BlueMind. BlueMind is a messaging and collaborative
* solution.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of either the GNU Affero General Public License as
* published by the Free Software Foundation (version 3 of the License).
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*
* See LICENSE.txt
* END LICENSE
*/

package net.bluemind.core.container.repository;

import java.sql.SQLException;
import java.util.Date;

import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.ItemIdentifier;
import net.bluemind.core.container.model.ItemVersion;

public interface IChangelogStore {

	public static class LogEntry {
		public long version;
		public String itemUid;
		public Date date;
		public long internalId;
		public long weightSeed;

		public static LogEntry create(long version, String itemUid, String itemExtId, String author, String origin,
				long itemId, long weightSeed) {
			LogEntry ret = new LogEntry();
			ret.version = version;
			ret.itemUid = itemUid;
			ret.internalId = itemId;
			ret.weightSeed = weightSeed;
			return ret;
		}

		public static LogEntry create(String itemUid, String itemExtId, String author, String origin, long itemId,
				long weightSeed) {
			return create(-1, itemUid, itemExtId, author, origin, itemId, weightSeed);
		}

		public static LogEntry create(long version, String itemUid, long itemId, long weightSeed) {
			LogEntry ret = new LogEntry();
			ret.version = version;
			ret.itemUid = itemUid;
			ret.internalId = itemId;
			ret.weightSeed = weightSeed;
			return ret;
		}
	}

	void itemCreated(LogEntry entry) throws SQLException;

	void itemUpdated(LogEntry entry) throws SQLException;

	void itemDeleted(LogEntry entry) throws SQLException;

	default ContainerChangeset<String> changeset(long from, long to) throws SQLException {
		return changeset(s -> s, from, to);
	}

	ContainerChangeset<String> changeset(IWeightProvider wp, long from, long to) throws SQLException;

	default ContainerChangeset<Long> changesetById(long from, long to) throws SQLException {
		return changesetById(s -> s, from, to);
	}

	ContainerChangeset<Long> changesetById(IWeightProvider wp, long from, long to) throws SQLException;

	default ContainerChangeset<ItemVersion> changesetById(long from, long to, ItemFlagFilter filter)
			throws SQLException {
		return changesetById(s -> s, from, to, filter);
	}

	ContainerChangeset<ItemVersion> changesetById(IWeightProvider wp, long from, long to, ItemFlagFilter filter)
			throws SQLException;

	ContainerChangeset<ItemIdentifier> fullChangesetById(IWeightProvider wp, long from, long to) throws SQLException;

	void deleteLog() throws SQLException;

	void allItemsDeleted(String subject, String origin) throws SQLException;

}