/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.filehosting.service.internal;

import static net.bluemind.filehosting.service.internal.PathValidator.validate;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.slf4j.LoggerFactory;

import com.google.common.annotations.VisibleForTesting;

import io.vertx.core.eventbus.EventBus;
import net.bluemind.core.api.Stream;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.domain.api.IDomainSettings;
import net.bluemind.eclipse.common.RunnableExtensionLoader;
import net.bluemind.filehosting.api.Configuration;
import net.bluemind.filehosting.api.FileHostingInfo;
import net.bluemind.filehosting.api.FileHostingInfo.Type;
import net.bluemind.filehosting.api.FileHostingItem;
import net.bluemind.filehosting.api.FileHostingPublicLink;
import net.bluemind.filehosting.api.IFileHosting;
import net.bluemind.filehosting.service.export.IFileHostingService;
import net.bluemind.filehosting.service.export.IInternalFileHostingService;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.sds.sync.api.SdsSyncEvent;
import net.bluemind.system.api.GlobalSettingsKeys;
import net.bluemind.system.api.IGlobalSettings;

public class FileHostingService implements IFileHosting {
	public final List<IFileHostingService> delegates;
	private final SecurityContext context;
	private final String domainUid;
	private final EventBus eventbus;
	private final IInternalFileHostingService defaultDelegate;
	private static final int SERVICE_UNAVAILABILITY = 10;
	@VisibleForTesting
	public static long serviceUnavailability = TimeUnit.MINUTES.toMillis(SERVICE_UNAVAILABILITY);
	private static Map<String, Long> blockedServices = new ConcurrentHashMap<>();

	public FileHostingService(SecurityContext context, String domainUid) {
		this.context = context;
		this.delegates = searchExtensionPoints();
		this.defaultDelegate = delegates.isEmpty() ? null : getInternalService();
		this.domainUid = domainUid;
		eventbus = VertxPlatform.eventBus();
	}

	private IInternalFileHostingService getInternalService() {
		for (IFileHostingService service : delegates) {
			if (service instanceof IInternalFileHostingService resolved) {
				if (resolved.supports(null)) {
					return resolved;
				}
			}
		}
		return null;
	}

	protected List<IFileHostingService> searchExtensionPoints() {
		RunnableExtensionLoader<IFileHostingService> epLoader = new RunnableExtensionLoader<>();
		List<IFileHostingService> extensions = epLoader.loadExtensions("net.bluemind.filehosting", "service", "service",
				"api");
		Collections.sort(extensions,
				(o1, o2) -> o1.isDefaultImplementation() ? 1 : o2.isDefaultImplementation() ? -1 : 0);

		return extensions;
	}

	@Override
	public List<FileHostingItem> list(String path) throws ServerFault {
		validate(path);
		return execute(context, service -> service.list(context, path));
	}

	@Override
	public List<FileHostingItem> find(String query) throws ServerFault {
		return execute(context, service -> service.find(context, query));
	}

	@Override
	public Stream get(String path) throws ServerFault {
		validate(path);
		return execute(context, service -> service.get(context, path));
	}

	@Override
	public FileHostingPublicLink share(String path, Integer downloadLimit, String expirationDate) throws ServerFault {
		validate(path);
		return execute(context, service -> service.share(context, path, downloadLimit, expirationDate));
	}

	@Override
	public FileHostingPublicLink storeAndShare(String path, Integer downloadLimit, String expirationDate,
			Stream document) throws ServerFault {
		validate(path);
		return execute(context, service -> {
			service.store(context, path, document);
			return service.share(context, path, downloadLimit, expirationDate);
		});
	}

	@Override
	public void unShare(String url) throws ServerFault {
		execute(context, service -> {
			service.unShare(context, url);
			return null;
		});
	}

	@Override
	public void store(String path, Stream document) throws ServerFault {
		validate(path);
		eventbus.publish(SdsSyncEvent.FHADD.busName(), path);
		execute(context, service -> {
			service.store(context, path, document);
			return null;
		});
	}

	@Override
	public void delete(String path) throws ServerFault {
		validate(path);
		execute(context, service -> {
			service.delete(context, path);
			return null;
		});
	}

	@Override
	public boolean exists(String path) throws ServerFault {
		validate(path);
		return execute(context, service -> service.exists(context, path));
	}

	@Override
	public FileHostingInfo info() throws ServerFault {
		FileHostingInfo info = new FileHostingInfo();
		if (delegates.isEmpty()) {
			info.present = false;
			return info;
		}
		int externalSystems = delegates.stream().map(d -> d.info(context).type)
				.filter(t -> t == null || t == Type.EXTERNAL).collect(Collectors.toList()).size();
		info.type = externalSystems > 0 ? Type.EXTERNAL : Type.INTERNAL;
		info.info = delegates.stream().map(d -> d.info(context).info).reduce("",
				(sum, infoString) -> sum.concat(infoString).concat("\n"));
		info.browsable = delegates.stream().map(d -> d.info(context).browsable).reduce(false,
				(browsable, browsableValue) -> browsable || browsableValue);
		info.present = true;
		return info;
	}

	@Override
	public Configuration getConfiguration() throws ServerFault {
		IGlobalSettings settingsGlobal = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IGlobalSettings.class);
		Map<String, String> values = settingsGlobal.get();

		IDomainSettings settingsDomain = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IDomainSettings.class, domainUid);
		Map<String, String> valuesDomain = settingsDomain.get();

		values.putAll(valuesDomain);

		Configuration config = new Configuration();
		config.maxFilesize = longValue(values, GlobalSettingsKeys.filehosting_max_filesize.name(), 0);
		config.retentionTime = longValue(values, GlobalSettingsKeys.filehosting_retention.name(), 365).intValue();
		return config;
	}

	private Long longValue(Map<String, String> map, String key, long defaultValue) {
		String value = map.get(key);
		if (value == null) {
			return defaultValue;
		} else {
			return Long.valueOf(value);
		}
	}

	private <V> V execute(SecurityContext ctx, Function<IFileHostingService, V> op) {
		for (IFileHostingService impl : delegates) {
			try {
				if (ctx == null || (!isExternalAccountBlocked(ctx, impl.info(ctx).type) && impl.supports(ctx))) {
					return op.apply(impl);
				}
			} catch (Exception e) {
				blockExternalAccount(ctx, impl.info(ctx).type);
				LoggerFactory.getLogger(FileHostingService.class)
						.warn("Filehosting operation failed using implementation {}", impl.info(ctx), e);
				throw e;
			}
		}
		throw new ServerFault(
				"All " + delegates.size() + " filehosting services failed to execute requested operation");
	}

	private void blockExternalAccount(SecurityContext ctx, Type type) {
		if (type == Type.EXTERNAL && ctx != null) {
			blockedServices.put(ctx.getSubject(), System.currentTimeMillis() + serviceUnavailability);
		}
	}

	private boolean isExternalAccountBlocked(SecurityContext ctx, Type type) {
		if (type == Type.INTERNAL) {
			return false;
		}
		if (!blockedServices.containsKey(ctx.getSubject())) {
			return false;
		} else {
			if (System.currentTimeMillis() > blockedServices.get(ctx.getSubject())) {
				blockedServices.remove(ctx.getSubject());
				return false;
			} else {
				return true;
			}
		}

	}

	public IInternalFileHostingService getDefaultDelegate() {
		return defaultDelegate;
	}

}
