/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2019
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.aws.s3.utils;

import java.net.URI;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Strings;
import com.typesafe.config.Config;

import io.netty.channel.epoll.Epoll;
import io.netty.channel.epoll.EpollEventLoopGroup;
import io.netty.channel.kqueue.KQueue;
import io.netty.channel.kqueue.KQueueEventLoopGroup;
import net.bluemind.configfile.core.CoreConfig;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.StaticCredentialsProvider;
import software.amazon.awssdk.core.checksums.RequestChecksumCalculation;
import software.amazon.awssdk.core.checksums.ResponseChecksumValidation;
import software.amazon.awssdk.http.SdkHttpConfigurationOption;
import software.amazon.awssdk.http.nio.netty.NettyNioAsyncHttpClient;
import software.amazon.awssdk.http.nio.netty.NettyNioAsyncHttpClient.Builder;
import software.amazon.awssdk.http.nio.netty.SdkEventLoopGroup;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.LegacyMd5Plugin;
import software.amazon.awssdk.services.s3.S3AsyncClient;
import software.amazon.awssdk.services.s3.S3AsyncClientBuilder;
import software.amazon.awssdk.utils.AttributeMap;

public class S3ClientFactory {
	@SuppressWarnings("serial")
	private static class S3ConfigException extends RuntimeException {

		public S3ConfigException(Exception e) {
			super(e);
		}

	}

	private static final Logger logger = LoggerFactory.getLogger(S3ClientFactory.class);

	private S3ClientFactory() {
	}

	public static S3AsyncClient create(S3Configuration s3Configuration) {
		Config coreConfig = CoreConfig.get();
		try {
			Builder nettySetup = NettyNioAsyncHttpClient.builder() //
					.maxConcurrency(coreConfig.getInt(CoreConfig.SDS.S3_MAX_CONNECTIONS)) //
					.tcpKeepAlive(coreConfig.getBoolean(CoreConfig.SDS.S3_TCP_KEEPALIVE)) //
					.connectionMaxIdleTime(coreConfig.getDuration(CoreConfig.SDS.S3_MAX_IDLE_TIME)) //
					.readTimeout(coreConfig.getDuration(CoreConfig.SDS.READ_TIMEOUT).minusSeconds(10)) //
					.writeTimeout(coreConfig.getDuration(CoreConfig.SDS.WRITE_TIMEOUT).minusSeconds(10)) //
					.connectionAcquisitionTimeout(coreConfig.getDuration(CoreConfig.SDS.S3_ACQUISITION_TIMEOUT));
			if (Epoll.isAvailable()) {
				nettySetup.eventLoopGroup(SdkEventLoopGroup.create(new EpollEventLoopGroup()));
			} else if (KQueue.isAvailable()) {
				nettySetup.eventLoopGroup(SdkEventLoopGroup.create(new KQueueEventLoopGroup()));
			} else {
				throw new UnsupportedOperationException("NO: neither epoll or kqueue are available");
			}

			S3AsyncClientBuilder builder = S3AsyncClient.builder();
			builder.httpClient(nettySetup.buildWithDefaults(AttributeMap.builder()
					.put(SdkHttpConfigurationOption.TRUST_ALL_CERTIFICATES, s3Configuration.isInsecure()).build()));
			builder.credentialsProvider(StaticCredentialsProvider.create(
					AwsBasicCredentials.create(s3Configuration.getAccessKey(), s3Configuration.getSecretKey())));
			if (!Strings.isNullOrEmpty(s3Configuration.getRegion())) {
				logger.info("Setting AWS Region to '{}'", s3Configuration.getRegion());
				builder.region(Region.of(s3Configuration.getRegion()));
			} else {
				builder.region(Region.AWS_GLOBAL);
			}
			builder.endpointOverride(new URI(s3Configuration.getEndpoint()));
			builder.serviceConfiguration(software.amazon.awssdk.services.s3.S3Configuration.builder()//
					.pathStyleAccessEnabled(true).build());
			builder.addPlugin(LegacyMd5Plugin.create())
					.requestChecksumCalculation(RequestChecksumCalculation.WHEN_REQUIRED)
					.responseChecksumValidation(ResponseChecksumValidation.WHEN_REQUIRED);
			return builder.build();
		} catch (Exception e) {
			throw new S3ConfigException(e);
		}
	}
}
