/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.lmtp.filter.tnef;

import java.util.Arrays;
import java.util.Collections;
import java.util.function.BiFunction;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.authentication.api.IAuthentication;
import net.bluemind.authentication.api.LoginResponse;
import net.bluemind.config.Token;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.serviceprovider.SPResolver;
import net.bluemind.user.api.IUser;
import net.bluemind.user.api.User;

public final class Sudo implements AutoCloseable {
	private static final Logger logger = LoggerFactory.getLogger(Sudo.class);

	private final ItemValue<User> user;
	private final String latd;
	public final SecurityContext context;

	private String authKey;

	private Sudo(ItemValue<User> user, String domainContainerUid) throws ServerFault {
		this.user = user;
		logger.debug("[{}] sudo login of uid {}, login {}", domainContainerUid, user.uid, user.value.login);

		LoginResponse auth = SPResolver.get().resolve(Token.admin0()).instance(IAuthentication.class)
				.su(user.value.login + "@" + domainContainerUid);
		this.latd = auth.latd;
		this.authKey = auth.authKey;
		SecurityContext userContext = new SecurityContext(authKey, user.uid, Arrays.<String>asList(),
				Arrays.<String>asList(), Collections.emptyMap(), domainContainerUid, "en", "Sudo", false);
		this.context = userContext;
	}

	public static Sudo byEmail(String email, String domain) {
		return sudo(email, domain, (service, input) -> service.byEmail(email));
	}

	public static Sudo byUid(String uid, String domain) {
		return sudo(uid, domain, (service, input) -> service.getComplete(uid));
	}

	private static Sudo sudo(String uid, String domain, BiFunction<IUser, String, ItemValue<User>> lookup) {
		IUser service = SPResolver.get().resolve(Token.admin0()).instance(IUser.class, domain);
		ItemValue<User> user = lookup.apply(service, uid);
		if (user == null) {
			throw ServerFault.notFound(uid + " in " + domain + " not found.");
		}
		return new Sudo(user, domain);

	}

	public String getLatd() {
		return latd;
	}

	public String getAuthKey() {
		return authKey;
	}

	public ItemValue<User> getUser() {
		return user;
	}

	public IServiceProvider provider() {
		return SPResolver.get().resolve(context.getSessionId());
	}

	@Override
	public void close() {
		provider().instance(IAuthentication.class).logout();
	}

}
