/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.validation.impl;

import java.net.URL;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;

import com.google.common.base.Throwables;

import net.bluemind.eas.validation.Activator;

public class ValidatorFactory {

	private static final Logger logger = LoggerFactory.getLogger(ValidatorFactory.class);

	public static enum Direction {
		request, response;
	}

	public static enum ProtoVersion {
		v12(12.1), v14(14.0), v16(16.1);

		private final double v;

		private ProtoVersion(double v) {
			this.v = v;
		}
	}

	private static String entryPath(ProtoVersion v, Direction d, String command) {
		StringBuilder sb = new StringBuilder();
		sb.append("xsd/").append(v.v).append("/").append(d.name()).append("/");
		switch (d) {
		case request:
			sb.append(command).append("Request.xsd");
			break;
		case response:
			sb.append(command).append("Response.xsd");
			break;
		}
		return sb.toString();
	}

	public static Validator createFor(ProtoVersion v, Direction d, String command) {
		if (command == null) {
			throw new RuntimeException("Command can't be null");
		}
		SchemaFactory sf = schemaFactory(v);
		String entryPath = entryPath(v, d, command);
		logger.debug("Loading schema {}", entryPath);
		URL documentSchema = Activator.getContext().getBundle().getEntry(entryPath);
		if (documentSchema == null) {
			logger.warn("Don't know how to validate {} {} {}", v, d, command);
			return null;
		}
		try {
			Source s = new StreamSource(documentSchema.openStream());
			Schema schema = sf.newSchema(s);
			return schema.newValidator();
		} catch (Exception e) {
			logger.error(e.getMessage(), e);
			throw Throwables.propagate(e);
		}
	}

	private static SchemaFactory schemaFactory(ProtoVersion v) {
		SchemaFactory sf = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
		Enumeration<URL> schemaUrls = Activator.getContext().getBundle().findEntries("xsd/" + v.v + "/", "*.xsd", true);
		List<URL> urlsList = new LinkedList<URL>();
		final Map<String, URL> sysIdToUrlIndex = new HashMap<>();
		while (schemaUrls.hasMoreElements()) {
			URL xsd = schemaUrls.nextElement();
			String str = xsd.toString();
			String sysId = str.substring(str.lastIndexOf('/') + 1);
			urlsList.add(xsd);
			sysIdToUrlIndex.put(sysId, xsd);
		}
		logger.debug("Prepared for resolving with {} xsd files.", urlsList.size());
		LSResourceResolver resolver = new LSResourceResolver() {

			@Override
			public LSInput resolveResource(String type, String namespaceURI, String publicId, String systemId,
					String baseURI) {
				logger.debug("resolve {}", systemId);
				URL u = sysIdToUrlIndex.get(systemId);
				LSInput ret = new BundleInput(u, type, namespaceURI, publicId, systemId, baseURI);
				return ret;
			}
		};
		sf.setResourceResolver(resolver);
		return sf;
	}

}
