/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.milter.impl;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MilterConnection implements Runnable {
	private final SocketChannel clientChannel;
	private Selector clientSelector;
	private final Map<SocketChannel, Selector> clientSelectors;
	private final AtomicBoolean running;
	private MilterSession session;
	String remoteAddress;
	private static final Logger logger = LoggerFactory.getLogger(MilterConnection.class);

	public MilterConnection(SocketChannel clientChannel, Map<SocketChannel, Selector> clientSelectors,
			AtomicBoolean running) {
		this.clientChannel = clientChannel;
		this.clientSelectors = clientSelectors;
		this.running = running;
	}

	@Override
	public void run() {
		try {
			clientSelector = Selector.open();
			clientSelectors.put(clientChannel, clientSelector);

			this.remoteAddress = clientChannel.getRemoteAddress().toString();
			session = new MilterSession(clientChannel, new MilterHandler(MLRegistry.getFactories()), clientSelector,
					remoteAddress);

			clientChannel.register(clientSelector, SelectionKey.OP_READ);

			logger.info("Starting client session for {}", remoteAddress);

			mainloop: while (running.get() && clientChannel.isConnected()) {
				if (clientSelector.select(1000) > 0) {
					Set<SelectionKey> selectedKeys = clientSelector.selectedKeys();
					Iterator<SelectionKey> iterator = selectedKeys.iterator();

					while (iterator.hasNext()) {
						SelectionKey key = iterator.next();
						iterator.remove();

						if (key.isReadable()) {
							if (!handleRead()) {
								break mainloop;
							}
						} else if (key.isWritable()) {
							handleWrite(key);
						}
					}
				}
			}
		} catch (IOException e) {
			logger.warn("Error handling client", e);
		} finally {
			logger.info("Ending client session for {}", remoteAddress);
			cleanup();
		}
	}

	private boolean handleRead() {
		try {
			ByteBuffer cmdBuffer = session.read();
			if (cmdBuffer != null) {
				session.processCommands(cmdBuffer);
			} else {
				return false;
			}
		} catch (Exception e) {
			logger.error("Read error, closing session, client disconnected: {} ", e.getMessage());
			return false;
		}
		return true;
	}

	private void handleWrite(SelectionKey key) {
		try {
			if (session.write()) {
				// All pending data written, stop listening for write events
				key.interestOps(SelectionKey.OP_READ);
			}
		} catch (IOException e) {
			logger.error("Write error, closing session: {}", e.getMessage());
		}
	}

	private void cleanup() {
		try {
			clientSelectors.remove(clientChannel);

			if (clientSelector != null) {
				clientSelector.close();
			}

			if (clientChannel.isOpen()) {
				clientChannel.close();
			}
		} catch (IOException e) {
			logger.warn("Error during cleanup: ", e);
		}
	}
}