package net.bluemind.indexing.incremental.hierarchy;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Stopwatch;

import net.bluemind.calendar.api.VEventSeries;
import net.bluemind.core.container.api.ContainerHierarchyNode;
import net.bluemind.core.container.api.IContainersFlatHierarchy;
import net.bluemind.core.container.api.IFlatHierarchyUids;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.ContainerChangeset;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.eclipse.common.RunnableExtensionLoader;
import net.bluemind.indexing.incremental.AbstractTypeIndexerByUid.UidIndexWriter;
import net.bluemind.indexing.incremental.TypeIndexerFactory;
import net.bluemind.indexing.incremental.hooks.IIncrementalIndexingContainersHook;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore;
import net.bluemind.indexing.incremental.repository.IIncrementalIndexingStore.ContainerSyncState;

public class FlatHierarchyIncrementalIndexFactory implements TypeIndexerFactory {

	private static final List<IIncrementalIndexingContainersHook> cHooks = loadContainerHooks();

	public static List<IIncrementalIndexingContainersHook> get() {
		return cHooks;
	}

	private static List<IIncrementalIndexingContainersHook> loadContainerHooks() {
		RunnableExtensionLoader<IIncrementalIndexingContainersHook> rel = new RunnableExtensionLoader<>();
		return List
				.copyOf(rel.loadExtensions("net.bluemind.core.container.hooks", "incrementalindexing", "hook", "impl"));
	}

	private static final Logger logger = LoggerFactory.getLogger(FlatHierarchyIncrementalIndexFactory.class);

	@Override
	public String type() {
		return IFlatHierarchyUids.TYPE;
	}

	@Override
	public TypeIndexer create(BmContext ctx, Container c) {

		IContainersFlatHierarchy flatHierarchyApi = ctx.provider().instance(IContainersFlatHierarchy.class, c.domainUid,
				c.owner);

		return new FlatHierarchyIndexer(flatHierarchyApi, new VEventSeriesIndexWriter(ctx));
	}

	public static class VEventSeriesIndexWriter implements UidIndexWriter<VEventSeries> {

		private BmContext ctx;

		public VEventSeriesIndexWriter(BmContext ctx) {
			this.ctx = ctx;
		}

		@Override
		public void write(ItemValue<VEventSeries> toIndex) {
			//
		}

		@Override
		public void deleteByUid(String uid, String type) {
			cHooks.forEach(h -> h.onContainerDeleted(ctx, uid, type));

		}

	}

	public static class FlatHierarchyIndexer implements TypeIndexerFactory.TypeIndexer {

		private IContainersFlatHierarchy clSupport;
		private UidIndexWriter<VEventSeries> indexer;

		public FlatHierarchyIndexer(IContainersFlatHierarchy rec, UidIndexWriter<VEventSeries> writer) {
			this.clSupport = rec;
			this.indexer = writer;
		}

		@Override
		public long indexDelta(ContainerSyncState startState) {
			if (startState.version() == IIncrementalIndexingStore.DO_NOT_INDEX) {
				return startState.version();
			}

			try {
				var chrono = Stopwatch.createStarted();
				ContainerChangeset<String> changeset = clSupport.changeset(startState.version());

				for (String deleted : changeset.deleted) {
					int l = deleted.indexOf(":");
					String type = deleted.substring(0, l);
					String cuid = ContainerHierarchyNode.extractContainerUid(deleted);
					indexer.deleteByUid(cuid, type);
				}
				if (logger.isInfoEnabled() && changeset.version > startState.version()) {
					logger.info("[{}] incr update from v{} to v{}  took {}ms (c: {}, u: {}, d: {})",
							startState.cont().value(), startState.version(), changeset.version,
							chrono.elapsed(TimeUnit.MILLISECONDS), changeset.created.size(), changeset.updated.size(),
							changeset.deleted.size());
				}
				return changeset.version;
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
				return startState.version();
			}
		}

	}

}
