/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.notes.impl;

import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.dataprotect.common.backup.JsonReadWrite;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.notes.api.VNote;

public class NotesBackupRepository {
	public static final Path DEFAULT_PATH = Paths.get("/var/backups/bluemind/notes");
	private static Logger logger = LoggerFactory.getLogger(NotesBackupRepository.class);
	RepositoryBackupPath repositoryPath;

	public NotesBackupRepository(RepositoryBackupPath repositoryPath) {
		this.repositoryPath = repositoryPath;
	}

	public NotesBackupRepository(Path repositoryPath) {
		this(new RepositoryBackupPath(repositoryPath));
	}

	private List<RestorableNotes> readZipFilesForUserNotes(String domainUid, String ownerUid) throws Exception {
		Optional<Path> zipFilePath = repositoryPath.getZipFilePathForOwner(domainUid, ownerUid);
		if (zipFilePath.isEmpty()) {
			return Collections.emptyList();
		}
		return extractUserNotesDataFromZip(ownerUid, zipFilePath.get());
	}

	private List<RestorableNotes> readJsonFilesForUserNotes(String ownerUid) throws IOException {
		Path restorablesPath = repositoryPath.resolveUserPath(ownerUid);
		return (List<RestorableNotes>) JsonReadWrite.readJsonFile(JsonUtils.reader(RestorableNotes.class),
				restorablesPath);
	}

	/**
	 * Returns the list of notes available in the backup
	 *
	 * @param ownerUid
	 * @return List of RestorableNotes
	 * @throws IOException
	 */
	public List<RestorableNotes> getRestorableNotes(String domainUid, String ownerUid) throws Exception {
		List<RestorableNotes> restoredNotes = readZipFilesForUserNotes(domainUid, ownerUid);
		if (restoredNotes.isEmpty()) {
			restoredNotes = readJsonFilesForUserNotes(ownerUid);
		}

		return restoredNotes;
	}

	record BackupNoteRecord(ContainerDescriptor cd, List<ItemValue<VNote>> notes) {

	}

	private String write(ContainerDescriptor containerDescriptor, List<ItemValue<VNote>> notes) throws IOException {
		Path jsonPathToWrite = repositoryPath.prepareJsonPathToWrite(containerDescriptor);
		JsonReadWrite.write(JsonUtils.writer(RestorableNotes.class), jsonPathToWrite,
				new RestorableNotes(containerDescriptor, notes));
		return RepositoryBackupPath.getFilePathWithDirectParent(jsonPathToWrite);
	}

	public List<String> writeNotes(List<BackupNoteRecord> backupNotes) throws IOException {
		List<String> ret = new ArrayList<>();
		for (BackupNoteRecord entryJson : backupNotes) {
			ret.add(write(entryJson.cd, entryJson.notes));
		}
		return ret;
	}

	private List<RestorableNotes> extractUserNotesDataFromZip(String folderName, Path zipFilePath) {
		List<RestorableNotes> notes = new ArrayList<>();
		try {
			JsonReadWrite.findJsonFilesWithPattern(folderName, zipFilePath).forEach(byteArray -> {
				try {
					notes.add((RestorableNotes) JsonReadWrite.readByteArr(JsonUtils.reader(RestorableNotes.class),
							byteArray));
				} catch (IOException e) {
					logger.error(e.getMessage());
				}
			});
		} catch (IOException e) {
			logger.error(e.getMessage());
		}

		return notes;
	}

}
