/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2026
 *
 * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License)
 * or the CeCILL as published by CeCILL.info (version 2 of the License).
 *
 * There are special exceptions to the terms and conditions of the
 * licenses as they are applied to this program. See LICENSE.txt in
 * the directory of this program distribution.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.lib.elasticsearch.client.transport;

import java.net.URI;
import java.time.Duration;
import java.util.List;
import java.util.Map;

import co.elastic.clients.json.JsonpMapper;
import co.elastic.clients.transport.ElasticsearchTransportBase;
import co.elastic.clients.transport.http.TransportHttpClient.Node;

public final class JdkHttpClientTransport extends ElasticsearchTransportBase {

	public static Builder builder(List<Node> nodes, JsonpMapper jsonpMapper) {
		return new Builder(nodes, jsonpMapper);
	}

	public static List<Node> nodesFromUris(String... uris) {
		return java.util.Arrays.stream(uris).map(uri -> new Node(URI.create(uri))).toList();
	}

	public static List<Node> nodesFromUris(List<String> uris) {
		return uris.stream().map(uri -> new Node(URI.create(uri))).toList();
	}

	private JdkHttpClientTransport(JdkHttpClient httpClient, JsonpMapper jsonpMapper) {
		super(httpClient, null, jsonpMapper);
	}

	public static class Builder {
		private final List<Node> nodes;
		private final JsonpMapper jsonpMapper;
		private int maxConnections = 128;
		private Duration connectTimeout = Duration.ofSeconds(10);
		private Duration requestTimeout = Duration.ofSeconds(30);
		private Duration connectionAcquireTimeout = Duration.ofSeconds(30);
		private Map<String, String> defaultHeaders = Map.of();

		private Builder(List<Node> nodes, JsonpMapper jsonpMapper) {
			this.nodes = nodes;
			this.jsonpMapper = jsonpMapper;
		}

		public Builder maxConnections(int maxConnections) {
			this.maxConnections = maxConnections;
			return this;
		}

		public Builder connectTimeout(Duration connectTimeout) {
			this.connectTimeout = connectTimeout;
			return this;
		}

		public Builder requestTimeout(Duration requestTimeout) {
			this.requestTimeout = requestTimeout;
			return this;
		}

		public Builder connectionAcquireTimeout(Duration connectionAcquireTimeout) {
			this.connectionAcquireTimeout = connectionAcquireTimeout;
			return this;
		}

		public Builder defaultHeaders(Map<String, String> headers) {
			this.defaultHeaders = headers;
			return this;
		}

		public JdkHttpClientTransport build() {
			JdkHttpClient httpClient = new JdkHttpClient(new NodePool(nodes), maxConnections, connectTimeout,
					requestTimeout, connectionAcquireTimeout, defaultHeaders);
			return new JdkHttpClientTransport(httpClient, jsonpMapper);
		}
	}
}
