/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2022
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.lib.elasticsearch;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.OptionalInt;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import co.elastic.clients.elasticsearch._types.ElasticsearchException;
import co.elastic.clients.elasticsearch.cat.AliasesResponse;
import net.bluemind.lib.elasticsearch.config.ElasticsearchClientConfig;
import net.bluemind.lib.elasticsearch.config.IndexAliasMode;
import net.bluemind.lib.elasticsearch.config.Mode;

public abstract class IndexAliasMapping {
	protected static final Logger logger = LoggerFactory.getLogger(IndexAliasMapping.class);

	public static final String MAILSPOOL_RING_ALIAS_READ_NAME = "mailspool_ring_alias_read";

	public abstract String getReadAliasByMailboxUid(String mailboxUid);

	public abstract String getIndexPrefix();

	public abstract String getAliasPrefix();

	public abstract String getWriteAliasByMailboxUid(String mailboxUid);

	public static IndexAliasMapping get() {
		if (IndexAliasMode.getMode() == Mode.TEST) {
			return new TestAliasMApping();
		}
		return IndexAliasMode.getMode() == Mode.ONE_TO_ONE ? new OneToOneIndexAliasMapping()
				: new RingIndexAliasMapping();
	}

	public static class TestAliasMApping extends IndexAliasMapping {
		private static final String MAILSPOOL_INDEX_PREFIX = "mailspool";
		private static final String MAILSPOOL_ALIAS_PREFIX = "mailspool_alias_";

		@Override
		public String getReadAliasByMailboxUid(String mailboxUid) {
			return getAlias(mailboxUid);
		}

		@Override
		public String getIndexPrefix() {
			return MAILSPOOL_INDEX_PREFIX;
		}

		@Override
		public String getAliasPrefix() {
			return MAILSPOOL_ALIAS_PREFIX;
		}

		@Override
		public String getWriteAliasByMailboxUid(String mailboxUid) {
			return getAlias(mailboxUid);
		}

		private String getAlias(String mailboxUid) {
			return MAILSPOOL_ALIAS_PREFIX + mailboxUid;
		}

	}

	public static class OneToOneIndexAliasMapping extends IndexAliasMapping {

		private static final String MAILSPOOL_INDEX_PREFIX = "mailspool_alias_";
		private static final String MAILSPOOL_ALIAS_PREFIX = "mailspool_alias_";

		@Override
		public String getReadAliasByMailboxUid(String mailboxUid) {
			return getAlias(mailboxUid);
		}

		@Override
		public String getWriteAliasByMailboxUid(String mailboxUid) {
			return getAlias(mailboxUid);
		}

		@Override
		public String getIndexPrefix() {
			return MAILSPOOL_INDEX_PREFIX;
		}

		@Override
		public String getAliasPrefix() {
			return MAILSPOOL_ALIAS_PREFIX;
		}

		private String getAlias(String mailboxUid) {
			return "mailspool_alias_" + mailboxUid;
		}

	}

	public static class RingIndexAliasMapping extends IndexAliasMapping {
		private static final Map<String, Integer> ALIAS_COUNT_MAP = new HashMap<>();
		private static final String MAILSPOOL_INDEX_PREFIX = "mailspool_ring_";
		private static final String MAILSPOOL_RING_ALIAS_PREFIX = "mailspool_ring_alias_";

		@Override
		public String getReadAliasByMailboxUid(String mailboxUid) {
			return MAILSPOOL_RING_ALIAS_READ_NAME + aliasPosition(mailboxUid);
		}

		@Override
		public String getWriteAliasByMailboxUid(String mailboxUid) {
			return "mailspool_ring_alias_write" + aliasPosition(mailboxUid);
		}

		@Override
		public String getIndexPrefix() {
			return MAILSPOOL_INDEX_PREFIX;
		}

		@Override
		public String getAliasPrefix() {
			return MAILSPOOL_RING_ALIAS_PREFIX;
		}

		private int aliasPosition(String mailboxUid) {
			int count = getMaxAliasCount();
			return (mailboxUid.hashCode() & 0xfffffff) % count;
		}

		private int getMaxAliasCount() {
			if (!ALIAS_COUNT_MAP.containsKey("mailspool")) {
				try {
					int aliasCount = getAliasCount();
					ALIAS_COUNT_MAP.put("mailspool", aliasCount);
				} catch (ElasticsearchException | IOException | NumberFormatException e) {
					logger.warn("Cannot return alias number for mailspool: use default value");
					return ESearchActivator.getIndexCount("mailspool")
							* ElasticsearchClientConfig.getMaxAliasMultiplier();
				}
			}
			return ALIAS_COUNT_MAP.get("mailspool");
		}

		private int getAliasCount() throws ElasticsearchException, IOException, NumberFormatException {
			AliasesResponse aliases = ESearchActivator.getClient().cat().aliases();
			OptionalInt maxAliasNumber = aliases.valueBody().stream()
					.filter(a -> a.alias().startsWith("mailspool_ring_alias_write"))
					.map(a -> a.alias().substring(a.alias().lastIndexOf("_write") + 6)). //
					mapToInt(a -> Integer.parseInt(a)) //
					.max();
			return maxAliasNumber.getAsInt() + 1;
		}

	}

}
