/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.sql.SQLException;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;
import com.datastax.oss.driver.api.core.uuid.Uuids;
import com.datastax.oss.driver.shaded.guava.common.collect.Lists;
import com.google.common.base.Splitter;
import com.google.common.base.Suppliers;

import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.replica.api.ImapBinding;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.api.MailboxRecord.InternalFlag;
import net.bluemind.backend.mail.replica.api.MailboxRecordItemUri;
import net.bluemind.backend.mail.replica.api.RawImapBinding;
import net.bluemind.backend.mail.replica.api.RecordID;
import net.bluemind.backend.mail.replica.api.WithId;
import net.bluemind.backend.mail.repository.IMailboxRecordStore;
import net.bluemind.core.container.api.Count;
import net.bluemind.core.container.cql.store.CqlContainerStore;
import net.bluemind.core.container.cql.store.CqlItemStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.CountFastPath;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.ItemFlag;
import net.bluemind.core.container.model.ItemFlagFilter;
import net.bluemind.core.container.model.SortDescriptor;
import net.bluemind.core.container.model.SortDescriptor.Direction;
import net.bluemind.cql.persistence.CqlAbstractStore;

public class CqlMailboxRecordStore extends CqlAbstractStore implements IMailboxRecordStore {

	private static final Logger logger = LoggerFactory.getLogger(CqlMailboxRecordStore.class);

	private final Container mbox;
	private final Container subtree;
	private final CqlBodyStore bodyStore;
	private final CqlMailboxRecordConversationStore mrConvStore;
	private final SortedRecordCqlStore sortedStore;
	private final LabelsCqlStore labelStore;

	public CqlMailboxRecordStore(CqlSession s, CqlContainerStore contStore, Container subtree, Container mbox,
			Item folderItem) {
		super(s);

		this.mbox = mbox;
		this.subtree = subtree;
		this.bodyStore = new CqlBodyStore(s);
		this.mrConvStore = new CqlMailboxRecordConversationStore(s, contStore, subtree);
		this.sortedStore = new SortedRecordCqlStore(s, subtree, mbox);
		this.labelStore = new LabelsCqlStore(s, subtree, folderItem);
	}

	private static final EntityPopulator<MailboxRecord> POP = (r, i, rec) -> {
		rec.imapUid = r.getLong(i++);
		rec.conversationId = r.getLong(i++);
		rec.internalDate = Date.from(r.getInstant(i++));
		rec.messageBody = r.getString(i++);
		rec.flags = new ArrayList<>(6);
		if (r.getBoolean(i++)) {
			rec.flags.add(MailboxItemFlag.System.Seen.value());
		}
		if (r.getBoolean(i++)) {
			rec.flags.add(MailboxItemFlag.System.Flagged.value());
		}
		if (r.getBoolean(i++)) {
			rec.flags.add(MailboxItemFlag.System.Draft.value());
		}
		if (r.getBoolean(i++)) {
			rec.flags.add(MailboxItemFlag.System.Answered.value());
		}
		if (r.getBoolean(i++)) {
			rec.flags.add(MailboxItemFlag.System.Deleted.value());
		}
		if (r.getBoolean(i++)) {
			rec.internalFlags.add(InternalFlag.expunged);
		}
		r.getList(i++, String.class).stream().map(MailboxItemFlag::new).forEach(rec.flags::add);

		return i;
	};
	private static final EntityCreator<MailboxRecord> ECR = r -> new MailboxRecord();

	@Override
	public void create(Item item, MailboxRecord value) throws SQLException {
		List<String> otherFlags = value.flags.stream().filter(mif -> mif.value == 0).map(mif -> mif.flag).toList();
		applyCql("""
				INSERT INTO t_mailbox_record
				(
				imap_uid, conversation_id, internal_date, message_body_guid,
				f_seen, f_flagged, f_draft, f_answered, f_deleted, f_expunged, other_flags,
				subtree_id, container_id, item_id
				)
				VALUES
				(?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
				""", value.imapUid, value.conversationId, value.internalDate.toInstant(), value.messageBody, //
				value.flags.contains(MailboxItemFlag.System.Seen.value()),
				value.flags.contains(MailboxItemFlag.System.Flagged.value()),
				value.flags.contains(MailboxItemFlag.System.Draft.value()),
				value.flags.contains(MailboxItemFlag.System.Answered.value()),
				value.flags.contains(MailboxItemFlag.System.Deleted.value()),
				value.internalFlags.contains(InternalFlag.expunged), otherFlags, //
				subtree.id, mbox.id, item.id);

		labelStore.addSome(Set.copyOf(otherFlags));

		Supplier<MessageBody> bodyProvider = Suppliers.memoize(() -> bodyStore.get(value.messageBody));
		bodyStore.refBody(value.messageBody, mbox, item);

		if (value.internalFlags.contains(InternalFlag.expunged)) {
			markExpunged(item, value);
		}
		if (value.internalFlags.contains(InternalFlag.expunged) || value.softDeleted()) {
			sortedStore.dropSortingArtifacts(item, value, bodyProvider);
		} else {
			sortedStore.writeSortingArtifacts(item, value, bodyProvider);
		}
		if (inConversation(value)) {
			if (value.internalFlags.contains(InternalFlag.expunged) || value.softDeleted()) {
				mrConvStore.deleteMessageRef(mbox, item, value);
			} else {
				// store internal msg refs
				logger.debug("writeInternalMessageRefs c: {} i: {}", mbox.id, item.id);
				mrConvStore.writeInternalMessageRefs(mbox, item, value, bodyProvider);
			}
		}
	}

	private boolean inConversation(MailboxRecord v) {
		return v.conversationId != null && v.conversationId != 0;
	}

	public void markExpunged(Item item, MailboxRecord value) {
		var created = Uuids.timeBased();
		int delta = Integer.parseInt(java.lang.System.getProperty("junit.expunged.days.delta", "0"));
		if (delta > 0) {
			created = timeBasedAt(delta);
		}
		applyCql("""
				INSERT INTO q_mailbox_record_expunged
				(global_id, subtree_id, container_id, created, item_id, imap_uid)
				VALUES
				(?, ?, ?, ?, ?, ?)
				""", 20120331L, subtree.id, mbox.id, created, item.id, value.imapUid);
		applyCql("""
				INSERT INTO q_mailbox_record_expunged_by_container
				(subtree_id, container_id, created, item_id, imap_uid)
				VALUES
				(?, ?, ?, ?, ?)
				""", subtree.id, mbox.id, created, item.id, value.imapUid);
	}

	private UUID timeBasedAt(int daysBefore) {
		long userProvidedTimestamp = Instant.now().minus(daysBefore, ChronoUnit.DAYS).toEpochMilli();
		Random random = ThreadLocalRandom.current();
		return new UUID(Uuids.startOf(userProvidedTimestamp).getMostSignificantBits(), random.nextLong());
	}

	@Override
	public void update(Item item, MailboxRecord value) throws SQLException {
		MailboxRecord prev = get(item);
		if (prev != null && !prev.messageBody.equals(value.messageBody)) {
			bodyStore.unrefBody(prev.messageBody, mbox, item);
		}
		sortedStore.dropPreviousSortingArtifacts(item);
		create(item, value);
	}

	@Override
	public void delete(Item item) throws SQLException {
		MailboxRecord forConvCheck = get(item);
		String del = """
				DELETE FROM t_mailbox_record
				WHERE subtree_id = ? AND container_id = ? AND item_id = ?
				""";
		voidCql(del, subtree.id, mbox.id, item.id);
		bodyStore.unrefBody(forConvCheck.messageBody, subtree, item);
		if (inConversation(forConvCheck)) {
			mrConvStore.deleteMessageRef(mbox, item, forConvCheck);
		}
		sortedStore.dropSortingArtifacts(item);
		String q = """
				SELECT created FROM q_mailbox_record_expunged_by_container
				WHERE subtree_id = ? AND container_id = ? AND item_id=?
				""";
		UUID timestamp = unique(q, r -> r.getUuid(0), (r, i, v) -> i, subtree.id, mbox.id, item.id);
		if (timestamp != null) {
			// clear expunged
			voidCql("DELETE FROM q_mailbox_record_expunged WHERE global_id=20120331 and created=?", timestamp);
			voidCql("DELETE FROM q_mailbox_record_expunged_by_container WHERE subtree_id = ? AND container_id = ? AND item_id=?",
					subtree.id, mbox.id, item.id);
		}
	}

	@Override
	public MailboxRecord get(Item item) throws SQLException {
		return unique("""
				SELECT
				imap_uid, conversation_id, internal_date, message_body_guid,
				f_seen, f_flagged, f_draft, f_answered, f_deleted, f_expunged, other_flags
				FROM t_mailbox_record
				WHERE subtree_id = ? AND container_id = ? AND item_id = ?
				""", ECR, POP, subtree.id, mbox.id, item.id);
	}

	@Override
	public void deleteAll() throws SQLException {
		String del = """
				DELETE FROM t_mailbox_record
				WHERE subtree_id = ? AND container_id = ?
				""";
		voidCql(del, subtree.id, mbox.id);
	}

	@Override
	public List<RecordID> identifiers(long... imapUids) throws SQLException {
		List<Long> fullList = Arrays.stream(imapUids).boxed().toList();
		List<RecordID> ret = new ArrayList<>(imapUids.length);
		for (var slice : Lists.partition(fullList, 64)) {
			List<RecordID> chunks = map("""
					SELECT
					item_id, imap_uid
					FROM t_mailbox_record_by_imap_uid
					WHERE subtree_id = ? AND container_id = ? AND imap_uid IN ?
					""", //
					r -> new RecordID(), //
					(r, i, rid) -> {
						rid.itemId = r.getLong(0);
						rid.imapUid = r.getLong(1);
						return i;
					}, subtree.id, mbox.id, slice);
			ret.addAll(chunks);
		}
		return ret;
	}

	@Override
	public List<ImapBinding> bindings(List<Long> itemIds) throws SQLException {
		List<ImapBinding> ret = new ArrayList<>(itemIds.size());
		for (var slice : Lists.partition(itemIds, 64)) {
			List<ImapBinding> chunk = map("""
					SELECT
					item_id, imap_uid, message_body_guid
					FROM t_mailbox_record
					WHERE subtree_id = ? AND container_id = ? AND item_id IN ?
					""", //
					r -> new ImapBinding(), //
					(r, i, rid) -> {
						rid.itemId = r.getLong(0);
						rid.imapUid = r.getLong(1);
						rid.bodyGuid = r.getString(2);
						return i;
					}, subtree.id, mbox.id, slice);
			ret.addAll(chunk);
		}
		return ret;
	}

	@Override
	public List<Long> sortedIds(boolean fast, SortDescriptor sortDesc) throws SQLException {
		logger.info("sortedIds: {}", sortDesc);
		String q = new CqlSortStrategy(sortDesc).queryToSort();
		List<Long> ret = map(q, r -> r.getLong(0), voidPop(), subtree.id, mbox.id);

		if (sortDesc.fields.stream().findFirst().map(f -> f.dir).orElse(Direction.Desc) == Direction.Asc) {
			Collections.reverse(ret);
		}
		return ret;
	}

	@Override
	public List<ImapBinding> havingBodyVersionLowerThan(int version) throws SQLException {
		List<ImapBinding> allBindings = map("""
				SELECT
				item_id, imap_uid, message_body_guid
				FROM t_mailbox_record
				WHERE subtree_id = ? AND container_id = ?
				""", //
				r -> new ImapBinding(), //
				(r, i, rid) -> {
					rid.itemId = r.getLong(0);
					rid.imapUid = r.getLong(1);
					rid.bodyGuid = r.getString(2);
					return i;
				}, subtree.id, mbox.id);
		List<String> bodies = allBindings.stream().map(ib -> ib.bodyGuid).toList();
		Set<String> lower = Set.copyOf(bodyStore.lowerThan(version, bodies));
		return allBindings.stream().filter(ib -> lower.contains(ib.bodyGuid)).toList();
	}

	@Override
	public List<ImapBinding> recentItems(Date d) throws SQLException {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<ImapBinding> listItemsAfter(Date d) throws SQLException {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public List<ImapBinding> unreadItems() throws SQLException {
		List<ToFilter> unfiltered = map("""
				SELECT
				item_id, imap_uid, f_seen, f_deleted, f_expunged, message_body_guid
				FROM t_mailbox_record
				WHERE subtree_id = ? AND container_id = ?
				""", //
				r -> new ToFilter(r.getLong(0), r.getLong(1), r.getBoolean(2), false, r.getBoolean(3), r.getBoolean(4),
						r.getString(5)),
				voidPop(), subtree.id, mbox.id);
		return unfiltered.stream().filter(t -> !t.seen() && !t.del() && !t.exp()).map(t -> {
			var ib = new ImapBinding();
			ib.imapUid = t.imapUid;
			ib.itemId = t.itemId;
			ib.bodyGuid = t.guid;
			return ib;
		}).toList();
	}

	@Override
	public Count count(ItemFlagFilter itemFilter) throws SQLException {
		if (itemFilter.matchDeleted()) {
			long cnt = unique("select count(*) from s_deleted_mailbox_record where subtree_id=? and container_id=?",
					r -> r.getLong(0), voidPop(), subtree.id, mbox.id);
			return Count.of(cnt);
		}

		long total = new CqlItemStore(mbox, session, null, null).count(itemFilter);
		return Count.of(total);
	}

	@Override
	public Optional<Count> fastpathCount(CountFastPath fastPath) {
		// TODO Auto-generated method stub
		return Optional.empty();
	}

	@Override
	public long weight() throws SQLException {
		// TODO Auto-generated method stub
		return 0;
	}

	@Override
	public List<MailboxRecordItemUri> getBodyGuidReferences(String guid) throws SQLException {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public String getImapUidReferences(long imapUid) throws SQLException {
		return unique("""
				SELECT
				message_body_guid
				FROM t_mailbox_record_by_imap_uid
				WHERE subtree_id = ? AND container_id = ? AND imap_uid = ?
				""", //
				r -> r.getString(0), //
				voidPop(), subtree.id, mbox.id, imapUid);
	}

	@Override
	public List<Long> getItemsByConversations(Long[] conversationIds) throws SQLException {
		return mrConvStore.getMessageRefsInFolder(mbox, Arrays.stream(conversationIds).toList()).stream()
				.map(imr -> imr.itemId).toList();
	}

	private record ToFilter(long itemId, long imapUid, boolean seen, boolean flagged, boolean del, boolean exp,
			String guid) {
	}

	@Override
	public List<RawImapBinding> imapIdset(String set, ItemFlagFilter itemFilter) throws SQLException {

		List<RawImapBinding> full = new ArrayList<>();
		String q = """
				SELECT
				item_id, imap_uid, f_seen, f_flagged, f_deleted, f_expunged
				FROM t_mailbox_record_by_imap_uid
				WHERE subtree_id = ? AND container_id = ?
				%s
				""";
		Predicate<ToFilter> filter = tf -> {
			if (tf.exp()) {
				return false;
			}
			if (itemFilter.must.contains(ItemFlag.Seen) && !tf.seen()) {
				return false;
			}
			if (itemFilter.must.contains(ItemFlag.Deleted) && !tf.del()) {
				return false;
			}
			if (itemFilter.mustNot.contains(ItemFlag.Seen) && tf.seen()) {
				return false;
			}
			if (itemFilter.mustNot.contains(ItemFlag.Deleted) && tf.del()) {
				return false;
			}
			return true;
		};
		List<String> allRanges = Splitter.on(',').splitToList(set);
		List<String> notRanges = allRanges.stream().filter(s -> !s.contains(":")).toList();
		List<String> onlyRanges = allRanges.stream().filter(s -> s.contains(":")).toList();
		if (!notRanges.isEmpty()) {
			for (var slice : Lists.partition(notRanges, 64)) {
				String inQ = slice.stream().collect(Collectors.joining(",", "AND imap_uid IN (", ")"));
				String cql = q.formatted(inQ);
				full.addAll(filterList(filter, cql));
			}
		}
		onlyRanges.forEach(s -> {
			String cql = q.formatted(imapUidFilter(s));
			full.addAll(filterList(filter, cql));
		});
		return full;
	}

	private List<RawImapBinding> filterList(Predicate<ToFilter> filter, String cql) {
		return map(cql,
				r -> new ToFilter(r.getLong(0), r.getLong(1), r.getBoolean(2), r.getBoolean(3), r.getBoolean(4),
						r.getBoolean(5), null),
				voidPop(), subtree.id, mbox.id).stream().filter(filter)
				.map(tf -> RawImapBinding.of(tf.imapUid, tf.itemId)).toList();
	}

	public static String imapUidFilter(String r) {
		if (r.equals("1:*")) {
			return "";
		}
		int idx = r.indexOf(':');
		if (idx > 0) {
			String start = r.substring(0, idx);
			String upper = r.substring(idx + 1, r.length());
			if (upper.equals("*")) {
				return "AND imap_uid >= " + start;
			} else {
				return "AND imap_uid >= " + start + " AND imap_uid <= " + upper;
			}
		} else {
			return "AND imap_uid = " + r;
		}
	}

	@Override
	public List<WithId<MailboxRecord>> slice(List<Long> itemIds) throws SQLException {
		List<WithId<MailboxRecord>> ret = new ArrayList<>(itemIds.size());
		for (var slice : Lists.partition(itemIds, 64)) {
			List<WithId<MailboxRecord>> chunk = map("""
					SELECT
					imap_uid, conversation_id, internal_date, message_body_guid,
					f_seen, f_flagged, f_draft, f_answered, f_deleted, f_expunged, other_flags,
					item_id
					FROM t_mailbox_record
					WHERE subtree_id = ? AND container_id = ? AND item_id IN ?
					""", //
					r -> new WithId<MailboxRecord>(), //
					(r, i, wid) -> {
						MailboxRecord rec = new MailboxRecord();
						int idx = POP.populate(r, i, rec);
						wid.itemId = r.getLong(idx++);
						wid.value = rec;
						return idx;
					}, subtree.id, mbox.id, slice);
			ret.addAll(chunk);
		}
		return ret;
	}

	@Override
	public List<WithId<MailboxRecord>> lightSlice(List<Long> itemIds) throws SQLException {
		return slice(itemIds);
	}

	@Override
	public List<String> labels() throws SQLException {
		return labelStore.getUnique();
	}

	@Override
	public boolean exists(Item item) throws SQLException {
		Boolean exists = unique(
				"select item_id from t_mailbox_record WHERE subtree_id = ? AND container_id = ? AND item_id=?",
				r -> true, voidPop(), subtree.id, mbox.id, item.id);
		return exists != null && exists.booleanValue();
	}

}
