/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.cql.store;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.backend.mail.api.Conversation;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.api.MessageBody.Recipient;
import net.bluemind.backend.mail.api.MessageBody.RecipientKind;
import net.bluemind.backend.mail.api.flags.MailboxItemFlag;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.InternalMessageRef;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.repository.IMailboxRecordConversationStore;
import net.bluemind.core.container.cql.store.CqlContainerStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.model.SortDescriptor;
import net.bluemind.core.container.model.SortDescriptor.Direction;
import net.bluemind.cql.persistence.CqlAbstractStore;

public class CqlMailboxRecordConversationStore extends CqlAbstractStore implements IMailboxRecordConversationStore {

	private static final Logger logger = LoggerFactory.getLogger(CqlMailboxRecordConversationStore.class);
	private final Container subtree;
	private CqlContainerStore contStore;

	public CqlMailboxRecordConversationStore(CqlSession s, CqlContainerStore store, Container subtree) {
		super(s);
		this.subtree = subtree;
		this.contStore = store;
	}

	private record VConv(String subject, String sender, int size, Instant first, Instant last, boolean unseen,
			boolean flagged) {

	}

	public void deleteMessageRef(Container mbox, Item it, MailboxRecord rec) {
		applyCql("""
				DELETE FROM t_conversion_message_refs
				WHERE subtree_id=? AND unique_id=? AND conversation_id=? AND item_id=?
				""", subtree.id, uniqueId(mbox), rec.conversationId, it.id);
		List<InternalMessageRef> remainingSiblings = getMessageRefsInFolder(mbox, List.of(rec.conversationId));
		if (remainingSiblings.isEmpty()) {
			// no trace of this conversation in the folder
			VConv curSortValues = vconv(mbox, rec.conversationId);
			if (curSortValues != null) {
				dropSortingArtifacts(mbox, rec, curSortValues);
			}
		} else {
			InternalMessageRef freshTopRef = remainingSiblings.stream()
					.sorted((mr1, mr2) -> mr2.date.compareTo(mr1.date)).findFirst().orElseThrow();
			List<RefFlags> remainingFlags = remainingSiblings.stream()
					.map(ir -> new RefFlags(ir.unseen, ir.flagged, mbox.uid)).toList();
			VConv curSortValues = vconv(mbox, rec.conversationId);
			if (curSortValues != null) {
				dropSortingArtifacts(mbox, rec, curSortValues);
				MailboxRecord fakeRec = new MailboxRecord();
				fakeRec.conversationId = rec.conversationId;
				fakeRec.internalDate = freshTopRef.date;
				MessageBody fakeBody = new MessageBody();
				fakeBody.recipients = List
						.of(Recipient.create(RecipientKind.Originator, freshTopRef.sender, freshTopRef.sender));
				fakeBody.size = freshTopRef.size;
				fakeBody.subject = freshTopRef.subject;
				fakeRec.flags = new ArrayList<>();
				boolean unseen = remainingFlags.stream().map(r -> r.unseen).reduce((a, b) -> a || b).orElseThrow();
				if (unseen) {
					fakeRec.flags.add(MailboxItemFlag.System.Seen.value());
				}
				boolean flagged = remainingFlags.stream().map(r -> r.flagged).reduce((a, b) -> a || b).orElseThrow();
				if (flagged) {
					fakeRec.flags.add(MailboxItemFlag.System.Flagged.value());
				}
				prepareSortArtifacts(mbox, fakeRec, fakeBody, freshTopRef.sender, null, null);
			}
		}
	}

	private void dropSortingArtifacts(Container mbox, MailboxRecord rec, VConv curSortValues) {
		batch(//
				b("DELETE FROM v_conversation_by_folder WHERE container_id=? AND conversation_id=?", mbox.id,
						rec.conversationId), //
				b("DELETE FROM s_conversation_by_folder_date_desc WHERE container_id=? AND date=? AND conversation_id=? ",
						mbox.id, curSortValues.last, rec.conversationId), //
				b("DELETE FROM s_conversation_by_folder_size_desc WHERE container_id=? AND size=? AND conversation_id=? ",
						mbox.id, curSortValues.size, rec.conversationId), //
				b("DELETE FROM s_conversation_by_folder_subject_desc WHERE container_id=? AND subject=? AND conversation_id=? ",
						mbox.id, curSortValues.subject, rec.conversationId), //
				b("DELETE FROM s_conversation_by_folder_sender_desc WHERE container_id=? AND sender=? AND conversation_id=? ",
						mbox.id, curSortValues.sender, rec.conversationId) //
		);
	}

	private static class RefFlags {
		boolean unseen;
		boolean flagged;
		String uniqueId;

		public RefFlags(boolean u, boolean f, String uniqueId) {
			this.unseen = u;
			this.flagged = f;
			this.uniqueId = uniqueId;
		}

	}

	public void writeInternalMessageRefs(Container mbox, Item it, MailboxRecord rec,
			Supplier<MessageBody> bodyProvider) {
		MessageBody body = bodyProvider.get();
		if (body == null) {
			logger.warn("Can't create v_conversation without a body {}", rec.messageBody);
			return;
		}
		String sender = body.recipients.stream().filter(r -> r.kind == RecipientKind.Originator)
				.map(r -> Optional.ofNullable(r.dn).orElse(r.address)).findFirst().orElse("");

		applyCql("""
				INSERT INTO t_conversion_message_refs
				(subtree_id, unique_id, conversation_id, item_id,
				internal_date, subject, sender, size, unseen, flagged)
				VALUES
				(?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
				""", subtree.id, IMailReplicaUids.getUniqueId(mbox.uid), rec.conversationId, it.id,
				rec.internalDate.toInstant(), SubjectCleaner.clean(body.subject), sender, body.size, rec.unseen(),
				rec.flagged());

		List<RefFlags> allFlags = map("""
				SELECT unseen, flagged, unique_id
				FROM t_conversion_message_refs
				WHERE subtree_id=? AND conversation_id=?
				""", r -> new RefFlags(r.getBoolean(0), r.getBoolean(1), r.getString(2)), voidPop(), subtree.id,
				rec.conversationId);

		VConv existing = vconv(mbox, rec.conversationId);

		VConv refreshed = prepareSortArtifacts(mbox, rec, body, sender, existing, allFlags);

		Set<String> otherFolders = allFlags.stream().map(r -> IMailReplicaUids.mboxRecords(r.uniqueId))
				.filter(f -> !f.equals(mbox.uid)).collect(Collectors.toSet());
		if (!otherFolders.isEmpty()) {
			propagateToOtherFolders(rec, refreshed, otherFolders);
		}

	}

	private void propagateToOtherFolders(MailboxRecord rec, VConv refreshed, Set<String> otherFolders) {
		for (String contUid : otherFolders) {
			Container otherBox = contStore.get(contUid);
			VConv existing = vconv(otherBox, rec.conversationId);
			insertSortingArtifacts(otherBox, rec.conversationId, existing, refreshed);
		}
	}

	private VConv prepareSortArtifacts(Container mbox, MailboxRecord rec, MessageBody body, String sender,
			VConv existing, List<RefFlags> allFlags) {
		Instant min;
		Instant max;
		int size;
		boolean unseen;
		boolean flagged;
		String subject;
		if (existing != null) {
			min = existing.first.isBefore(rec.internalDate.toInstant()) ? existing.first : rec.internalDate.toInstant();
			max = existing.last.isAfter(rec.internalDate.toInstant()) ? existing.last : rec.internalDate.toInstant();
			size = body.size > existing.size ? body.size : existing.size;
			unseen = allFlags.stream().map(r -> r.unseen).reduce((a, b) -> a || b).orElseThrow();
			flagged = allFlags.stream().map(r -> r.flagged).reduce((a, b) -> a || b).orElseThrow();
			subject = existing.subject;
		} else {
			subject = SubjectCleaner.clean(body.subject);
			min = rec.internalDate.toInstant();
			max = min;
			size = body.size;
			unseen = rec.unseen();
			flagged = rec.flagged();
		}
		VConv updated = new VConv(subject, sender, size, min, max, unseen, flagged);
		insertSortingArtifacts(mbox, rec.conversationId, existing, updated);
		return updated;
	}

	private void insertSortingArtifacts(Container mbox, long cid, VConv prev, VConv cur) {
		String insertQuery = """
				INSERT INTO v_conversation_by_folder
				(
				container_id, conversation_id,
				size, first, last,
				subject, last_sender, unseen, flagged
				)
				VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
				""";

		applyCql(insertQuery, mbox.id, cid, cur.size, cur.first, cur.last, cur.subject, cur.sender, cur.unseen,
				cur.flagged);

		if (prev != null) {
			// we need to delete mutable values from the sort tables
			batch(//
					b("DELETE FROM s_conversation_by_folder_date_desc WHERE container_id=? AND date=? AND conversation_id=?",
							mbox.id, prev.last, cid), //
					b("DELETE FROM s_conversation_by_folder_size_desc WHERE container_id=? AND size=? AND conversation_id=?",
							mbox.id, prev.size, cid), //
					b("DELETE FROM s_conversation_by_folder_sender_desc WHERE container_id=? AND sender=? AND conversation_id=?",
							mbox.id, prev.sender, cid)//
			);
		}
		batch(//
				b("""
						INSERT INTO s_conversation_by_folder_date_desc
						(container_id, conversation_id, date, unseen, flagged)
						VALUES (?,?,?,?,?)
						""", mbox.id, cid, cur.last, cur.unseen, cur.flagged), //
				b("""
						INSERT INTO s_conversation_by_folder_size_desc
						(container_id, conversation_id, size, unseen, flagged)
						VALUES (?,?,?,?,?)
						""", mbox.id, cid, cur.size, cur.unseen, cur.flagged), //
				b("""
						INSERT INTO s_conversation_by_folder_subject_desc
						(container_id, conversation_id, subject, unseen, flagged)
						VALUES (?,?,?,?,?)
						""", mbox.id, cid, cur.subject, cur.unseen, cur.flagged), //
				b("""
						INSERT INTO s_conversation_by_folder_sender_desc
						(container_id, conversation_id, sender, unseen, flagged)
						VALUES (?,?,?,?,?)
						""", mbox.id, cid, cur.sender, cur.unseen, cur.flagged) //
		);
	}

	private VConv vconv(Container mbox, long conversationId) {
		return unique("""
				SELECT subject, size, first, last, unseen, flagged, last_sender
				FROM v_conversation_by_folder
				WHERE container_id=? AND conversation_id=?
				""", r -> new VConv(r.getString(0), r.getString(6), r.getInt(1), r.getInstant(2), r.getInstant(3),
				r.getBoolean(4), r.getBoolean(5)), voidPop(), mbox.id, conversationId);
	}

	private static final EntityPopulator<InternalMessageRef> IMR_POP = (r, i, mr) -> {
		mr.folderUid = r.getUuid(i++).toString().replace("-", "");
		mr.conversationId = r.getLong(i++);
		mr.itemId = r.getLong(i++);
		mr.date = dateOrNull(r, i++);
		mr.subject = r.getString(i++);
		mr.sender = r.getString(i++);
		mr.size = r.getInt(i++);
		mr.unseen = r.getBoolean(i++);
		mr.flagged = r.getBoolean(i++);
		return i;
	};

	public List<InternalMessageRef> getMessageRefs(List<Long> conversationIds) {
		return map("""
				SELECT unique_id, conversation_id, item_id,
				internal_date, subject, sender, size, unseen, flagged
				FROM t_conversion_message_refs
				WHERE subtree_id=? AND conversation_id IN ?
				""", //
				r -> new InternalMessageRef(), IMR_POP, subtree.id, conversationIds);
	}

	public List<InternalMessageRef> getMessageRefsInFolder(Container mbox, List<Long> conversationIds) {
		return map("""
				SELECT unique_id, conversation_id, item_id,
				internal_date, subject, sender, size, unseen, flagged
				FROM t_conversion_message_refs
				WHERE subtree_id=? AND unique_id=? AND conversation_id IN ?
				""", //
				r -> new InternalMessageRef(), IMR_POP, subtree.id, uniqueId(mbox), conversationIds);
	}

	private String uniqueId(Container mbox) {
		return IMailReplicaUids.getUniqueId(mbox.uid);
	}

	public List<Long> getConversationIds(long folderId, SortDescriptor sorted) {
		StringBuilder query = new StringBuilder("SELECT conversation_id FROM ");
		query.append(chooseTable(sorted));
		query.append(" WHERE container_id = ?");
		if (sorted.isFilteredOnNotDeletedAndImportant()) {
			query.append(" AND flagged=true ALLOW FILTERING");
		} else if (sorted.isFilteredOnNotDeletedAndNotSeen()) {
			query.append(" AND unseen=true ALLOW FILTERING");
		}
		String q = query.toString();
		var ret = map(q, r -> r.getLong(0), voidPop(), folderId);
		if (logger.isDebugEnabled()) {
			logger.debug("conv sort in {}: {} -> {}", folderId, q, ret.size());
		}
		Direction dir = sorted.fields.stream().findFirst().map(f -> f.dir).orElse(SortDescriptor.Direction.Desc);
		if (dir == Direction.Asc) {
			Collections.reverse(ret);
		}
		return ret;

	}

	private String chooseTable(SortDescriptor sorted) {
		return sorted.fields.stream().findFirst().map(f -> {
			String r = "s_conversation_by_folder_";
			switch (f.column) {
			case "size":
				r += "size_desc";
				break;
			case "sender":
				r += "sender_desc";
				break;
			case "subject":
				r += "subject_desc";
				break;
			default:
			case "date", "internal_date", "first":
				r += "date_desc";
				break;
			}
			return r;
		}).orElse("s_conversation_by_folder_date_desc");
	}

	public Conversation get(String uid) {
		long convId = Long.parseUnsignedLong(uid, 16);
		return Conversation.of(getMessageRefs(List.of(convId)).stream().map(InternalMessageRef::toMessageRef).toList(),
				convId);
	}

	public List<Conversation> getMultiple(List<String> uid) {
		return uid.stream().map(this::get).toList();
	}

}